"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AdminValidatorAdapter = void 0;
const tslib_1 = require("tslib");
const typedi_1 = require("typedi");
const _1 = tslib_1.__importStar(require("."));
const db_types_1 = require("../../types/db.types");
const bcrypt = tslib_1.__importStar(require("bcryptjs"));
const enum_1 = require("../../types/enum");
const factory_1 = require("../../Utils/factory");
let AdminAdapter = class AdminAdapter extends _1.BaseAdapter {
    constructor() {
        super(...arguments);
        this.DBGetDashboardData = () => tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                //       const students = await knex(ETables.STUDENT).select('stream').groupBy("stream").count('* as student_count')
                //       const total_students = students.map((item: any) => {
                //         return { [item.stream]: parseInt(item.student_count) };
                //     });
                // console.log(total_students)
                // const total_students = await knex(ETables.STUDENT).count('* as student_count').groupBy("stream");
                const total_students = (0, _1.default)(db_types_1.ETables.STUDENT).count('* as student_count');
                // console.log(total_students)
                const prof_aspirants = (0, _1.default)(db_types_1.ETables.PROF_ASPIRANT).where('status', 'pending').count('* as professional_aspirants');
                const aspirants = (0, _1.default)(db_types_1.ETables.ASPIRANT).where('status', 'pending').count('* as aspirants_count');
                const courses = (0, _1.default)(db_types_1.ETables.COURSES).count('* as courses_count');
                const diets = (0, _1.default)(db_types_1.ETables.DIET).count('* as total_diet');
                const sessions = (0, _1.default)(db_types_1.ETables.DIET).count('* as total_sessions');
                const [[{ student_count }], [{ professional_aspirants }], [{ aspirants_count }], [{ courses_count }], [{ total_diet }], [{ total_sessions }]] = yield Promise.all([total_students, prof_aspirants, aspirants, courses, diets, sessions]);
                // console.log({ student_count, professional_aspirants, aspirants_count, courses_count })
                return { student_count, professional_aspirants, aspirants_count, courses_count, total_diet, total_sessions };
            }
            catch (error) {
                return this.catchError(error);
            }
        });
        this.DBCreateAdmin = (data) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                let [user] = yield _1.default.table(db_types_1.ETables.ADMIN).insert(data);
                return user;
            }
            catch (error) {
                return this.catchError(error);
            }
        });
        this.DBGetAdmins = () => tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                const admins = yield _1.default.select([
                    "admin.id as id",
                    "admin.email as email",
                    "admin.role_id as role_id",
                    "admin.name as name",
                    "role.title as role",
                    "role.permissions as permissions"
                ])
                    .from({ admin: db_types_1.ETables.ADMIN })
                    .join({ role: db_types_1.ETables.ADMIN_ROLE }, 'role.id', 'admin.role_id');
                return admins;
            }
            catch (error) {
                return this.catchError(error);
            }
        });
        this.DBEditAdmin = (id, data) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                yield (0, _1.default)(db_types_1.ETables.ADMIN).update(data).where('id', id);
            }
            catch (error) {
                return this.catchError(error);
            }
        });
        this.DBCreateSuperAdmin = () => tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                const password = bcrypt.hashSync(process.env.ADMIN_PASS, 10);
                yield _1.default.transaction((trx) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    let role = yield trx.select("*").from(db_types_1.ETables.ADMIN_ROLE).where("title", "Super Admin").first();
                    if (!role) {
                        [role] = yield trx(db_types_1.ETables.ADMIN_ROLE).insert({
                            title: "Super Admin",
                            description: "All priviledged admin",
                            permissions: enum_1.Permisions.SUDO
                        });
                        yield trx(db_types_1.ETables.ADMIN).insert({
                            password, email: "admin@tnmmediaacademy.com", role_id: role, name: "Super Admin",
                        });
                    }
                }));
            }
            catch (error) {
                return this.catchError(error);
            }
        });
        this.DBCreateAdminRole = (data) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                const [role] = yield _1.default.table(db_types_1.ETables.ADMIN_ROLE).insert(data);
                return role;
            }
            catch (error) {
                return this.catchError(error);
            }
        });
        this.DBGetRoles = () => tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                const roles = yield _1.default.select("*").from(db_types_1.ETables.ADMIN_ROLE).whereNot('permissions', '*');
                return roles;
            }
            catch (error) {
                return this.catchError(error);
            }
        });
        this.DBGetMe = (id) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                const me = yield _1.default.select([
                    'adm.*',
                    'role.title as role',
                    'role.permissions as permissions'
                ])
                    .from({ adm: db_types_1.ETables.ADMIN })
                    .join({ role: db_types_1.ETables.ADMIN_ROLE }, 'role.id', 'adm.role_id')
                    .where('adm.id', id).first();
                return me;
            }
            catch (error) {
                return this.catchError(error);
            }
        });
        this.DBGetSessions = () => tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                const sessions = yield _1.default.select('*').from(db_types_1.ETables.ACADEMIC_SESSION);
                return sessions;
            }
            catch (error) {
                return this.catchError(error);
            }
        });
        this.DBGetAllProgrammes = () => tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                const programmes = yield _1.default.select('*').from(db_types_1.ETables.PROGRAMME);
                return programmes;
            }
            catch (error) {
                return this.catchError(error);
            }
        });
        this.DBGetProgramme = (prog_level) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                const programme = yield _1.default.select('*').from(db_types_1.ETables.PROGRAMME).where("prog_level", prog_level).first();
                return programme;
            }
            catch (error) {
                return this.catchError(error);
            }
        });
        this.DBGetCourses = (stream) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                const courses = yield _1.default.select('*').from(db_types_1.ETables.COURSES).where("stream", stream);
                return courses;
            }
            catch (error) {
                return this.catchError(error);
            }
        });
        this.DBGetACourse = (stream, id) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                const course = yield _1.default.select('*').from(db_types_1.ETables.COURSES).where("stream", stream).andWhere('id', id).first();
                return course;
            }
            catch (error) {
                return this.catchError(error);
            }
        });
        this.DBGetAssignedCourses = (admin_id, stream) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                const assigned_courses = yield _1.default.select([
                    "ass.*",
                    "course.code as code",
                    "course.title as title",
                    "course.description as description",
                    "course.unit as unit",
                    "course.type as type",
                    "course.level as level",
                    "course.semester as semester",
                ])
                    .from({ ass: db_types_1.ETables.ASSIGNED_COURSES })
                    .join({ course: db_types_1.ETables.COURSES }, 'course.id', 'ass.course_id')
                    .where("ass.stream", stream).andWhere('admin_id', admin_id);
                return assigned_courses;
            }
            catch (error) {
                return this.catchError(error);
            }
        });
        this.DBAssignStaff = (stream, id, admin_ids, session_id) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                const obj = [];
                for (let admin_id of admin_ids) {
                    obj.push({
                        admin_id,
                        course_id: id,
                        stream,
                        session_id
                    });
                }
                yield (0, _1.default)(db_types_1.ETables.ASSIGNED_COURSES).insert(obj);
            }
            catch (error) {
                return this.catchError(error);
            }
        });
        this.DBGetAssignedStaff = (stream, id, session_id) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                const staff = yield _1.default.select([
                    "ass.*",
                    'admin.name as name',
                    'role.title as role'
                ])
                    .from({ ass: db_types_1.ETables.ASSIGNED_COURSES })
                    .join({ admin: db_types_1.ETables.ADMIN }, 'admin.id', 'ass.admin_id')
                    .join({ role: db_types_1.ETables.ADMIN_ROLE }, 'role.id', 'admin.role_id')
                    .where('stream', stream).andWhere('course_id', id).andWhere('session_id', session_id).first();
                return staff;
            }
            catch (error) {
                return this.catchError(error);
            }
        });
        this.DBGetStudents = () => tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                const students = yield _1.default.select([
                    "enrl.*",
                    "std.first_name as first_name",
                    "std.last_name as last_name",
                    "std.email as email",
                    "std.adm_no as adm_no",
                    "std.level as level",
                    // "std.phone as phone",
                    "std.DOB as DOB",
                    "prg.name as programme",
                    "prg.duration as duration",
                    "prg.stream as stream",
                    "diet.number as diet"
                ])
                    .from({ enrl: db_types_1.ETables.ENROLLMENT })
                    .join({ std: db_types_1.ETables.STUDENT }, "std.id", "enrl.student_id")
                    .leftJoin({ prg: db_types_1.ETables.PROGRAMME }, "prg.id", "enrl.programme_id")
                    .leftJoin({ diet: db_types_1.ETables.DIET }, "diet.id", "enrl.diet_id");
                console.log(students[0]);
                return students;
            }
            catch (error) {
                return this.catchError(error);
            }
        });
        this.DBGetPendingApplications = (programme) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                let applicatiions;
                // if (programme === Stream.STANDARD) {
                if (Object.values(enum_1.NID_Stream).includes(programme)) {
                    applicatiions = yield _1.default.select(['aspi.*'])
                        .from({ aspi: db_types_1.ETables.ASPIRANT })
                        .join({ prog: db_types_1.ETables.PROGRAMME }, 'prog.id', 'aspi.programme_id')
                        .where("status", enum_1.Status.PENDING).andWhere('prog.stream', programme);
                }
                else {
                    applicatiions = yield _1.default.select(['aspi.*'])
                        .from({ aspi: db_types_1.ETables.PROF_ASPIRANT })
                        .join({ prog: db_types_1.ETables.PROGRAMME }, 'prog.id', 'aspi.programme_id')
                        .where("status", enum_1.Status.PENDING).andWhere('prog.stream', programme);
                }
                return applicatiions;
            }
            catch (error) {
                return this.catchError(error);
            }
        });
        this.DBGetApplication = (id, programme) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                let applicatiions;
                const select = [
                    "std.*",
                    "prog.name as programme",
                    "prog.description as description",
                    "prog.duration as duration",
                    "prog.stream as stream",
                    "prog.open_to_application as open_to_application",
                ];
                // if (programme === Stream.STANDARD) {
                if (Object.values(enum_1.NID_Stream).includes(programme)) {
                    applicatiions = yield _1.default.select(select)
                        .from({ std: db_types_1.ETables.ASPIRANT })
                        .leftJoin({ prog: db_types_1.ETables.PROGRAMME }, 'prog.id', 'std.programme_id')
                        .where("status", enum_1.Status.PENDING).andWhere("std.id", id).first();
                }
                else {
                    applicatiions = yield _1.default.select(select)
                        .from({ std: db_types_1.ETables.PROF_ASPIRANT })
                        .join({ prog: db_types_1.ETables.PROGRAMME }, 'prog.id', 'std.programme_id')
                        .where("status", enum_1.Status.PENDING).andWhere("std.id", id).first();
                }
                return applicatiions;
            }
            catch (error) {
                return this.catchError(error);
            }
        });
        this.DBReviewApplication = (id, programme, review) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                // if (programme === Stream.STANDARD) 
                if (Object.values(enum_1.NID_Stream).includes(programme)) {
                    yield (0, _1.default)(db_types_1.ETables.ASPIRANT).update(review).where('id', id).onConflict().merge();
                }
                else {
                    yield (0, _1.default)(db_types_1.ETables.PROF_ASPIRANT).update(review).where('id', id).onConflict().merge();
                }
            }
            catch (error) {
                return this.catchError(error);
            }
        });
        this.DBGetCourseResult = (id, session_id) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                const results = yield _1.default.select([
                    "reg.*",
                    "std.first_name as first_name",
                    "std.last_name as last_name",
                    "std.email as email",
                    "std.adm_no as adm_no",
                    "std.matric_no as matric_no",
                ])
                    .from({ reg: db_types_1.ETables.COURSE_REG })
                    .join({ std: db_types_1.ETables.STUDENT }, 'std.id', 'reg.student_id')
                    .where("course_id", id).andWhere("session_id", session_id);
                return results;
            }
            catch (error) {
                return this.catchError(error);
            }
        });
        this.DBUploadResult = (student_id, score, course_id, session_id) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                yield (0, _1.default)(db_types_1.ETables.COURSE_REG).update({ grade: score }).where('student_id', student_id).andWhere({ course_id, session_id });
            }
            catch (error) {
                return this.catchError(error);
            }
        });
        this.DBGetSelectedStudents = (matric_nos) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                const students = yield _1.default.select('*').from(db_types_1.ETables.STUDENT).whereIn('matric_no', matric_nos);
                return students;
            }
            catch (error) {
                return this.catchError(error);
            }
        });
        this.DBUploadResultV2 = (updates, course_id, session_id, level) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                // let caseStatement = 'CASE';
                // updates.forEach(update => {
                //   caseStatement += knex.raw(` WHEN student_id = ? AND course_id = ? THEN ?`, [update.student_id, update.course_id, update.grade]).toString();
                // });
                // caseStatement += ' END';
                // let updateQuery = knex(ETables.COURSE_REG)
                //   .update({
                //     grade: knex.raw(caseStatement)
                //   })
                //   .where(knex.raw(`(student_id, course_id) IN (${updates.map(update => `(${update.student_id}, ${update.course_id})`).join(', ')})`))
                //   .toString();
                // knex.raw(updateQuery)
                //   .then(() => console.log("Batch update completed"))
                //   .catch(err => console.error(err));
                console.log(updates, 'updates');
                // Preparing the CASE statements for both ca and exam fields
                let caCaseStatement = 'CASE student_id ';
                let examCaseStatement = 'CASE student_id ';
                // Loop through each update to append conditions for both fields
                updates.forEach(update => {
                    caCaseStatement += _1.default.raw(`WHEN ? THEN ? `, [update.student_id, update.CA]);
                    examCaseStatement += _1.default.raw(`WHEN ? THEN ? `, [update.student_id, update.exam]);
                });
                // Closing each CASE statement
                caCaseStatement += 'ELSE grade END';
                examCaseStatement += 'ELSE exam END';
                // Executing the batch update with both CASE statements
                (0, _1.default)(db_types_1.ETables.COURSE_REG)
                    .update({
                    CA: _1.default.raw(caCaseStatement),
                    exam: _1.default.raw(examCaseStatement),
                    grade: _1.default.raw('ca + exam')
                })
                    .where({
                    session_id,
                    course_id,
                    current_level: level,
                })
                    .whereIn('student_id', updates.map(update => update.student_id))
                    .then(() => console.log("Batch update completed"))
                    .catch(err => console.error(err));
            }
            catch (error) {
                return this.catchError(error);
            }
        });
        this.DBGetAllFee = () => tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                const fees = yield _1.default.select('*').from(db_types_1.ETables.FEE);
                return fees;
            }
            catch (error) {
                return this.catchError(error);
            }
        });
        this.DBGetOtherFee = () => tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                const fees = yield _1.default.select('*').from(db_types_1.ETables.OTHER_FEE);
                return fees;
            }
            catch (error) {
                return this.catchError(error);
            }
        });
        this.DBSetFee = (data) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                const [fees] = yield (0, _1.default)(db_types_1.ETables.FEE).insert(Object.assign({}, data));
                return fees;
            }
            catch (error) {
                return this.catchError(error);
            }
        });
        this.DBUploadAspirants = (data, aspirants) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                // await knex.transaction(async trx => {
                //   const asprnts = [];
                //   for(let aspi of aspirants) {
                //     const obj = {
                //       jamb_reg: aspi.jamb_reg,
                //       email: aspi.email
                //     }
                //     asprnts.push(obj);
                //   };
                //   await trx(ETables.ASPIRANT).insert(aspirants);
                //   await trx(ETables.RESULT).insert({ ...data }); 
                // })
                const asprnts = [];
                for (let aspi of aspirants) {
                    const password = bcrypt.hashSync(aspi['Last Name'], 10);
                    const obj = {
                        jamb_reg: aspi['Jamb Registration'],
                        email: aspi['Email'],
                        first_name: aspi['First Name'],
                        last_name: aspi['Last Name'],
                        DOB: (0, factory_1.excelSerialDateToJSDate)(aspi.DOB),
                        password
                    };
                    asprnts.push(obj);
                }
                ;
                console.log(aspirants, 'exAspirants');
                yield (0, _1.default)(db_types_1.ETables.ASPIRANT).insert(asprnts);
            }
            catch (error) {
                return this.catchError(error);
            }
        });
        this.DBGetAspirants = () => tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                const aspirants = yield _1.default.select('*').from(db_types_1.ETables.ASPIRANT);
                return aspirants;
            }
            catch (error) {
                return this.catchError(error);
            }
        });
        this.DBViewAspirant = (jamb_reg) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                const aspirant = yield _1.default.select('*').from(db_types_1.ETables.ASPIRANT).where("jamb_reg", jamb_reg).first();
                return aspirant;
            }
            catch (error) {
                return this.catchError(error);
            }
        });
        this.DBUploadPutmeResult = (data, aspirants, results) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                yield _1.default.transaction((trx) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const reslt = yield trx.select('*').from(db_types_1.ETables.RESULT).where({ type: data.type, session_id: data.session_id, year: data.year }).first();
                    if (reslt) {
                        console.log('result exist ooo');
                        yield trx(db_types_1.ETables.RESULT).update({ file: data.file }).where({ type: data.type, session_id: data.session_id, year: data.year }).onConflict().merge();
                    }
                    else {
                        yield trx(db_types_1.ETables.RESULT).insert(Object.assign({}, data));
                    }
                    for (let aspirant of aspirants) {
                        for (let result of results) {
                            console.log('updating aspirant');
                            if (aspirant.jamb_reg === result.jamb_reg) {
                                yield trx(db_types_1.ETables.ASPIRANT).update({ grade: result.grade || result.score }).where('jamb_reg', aspirant.jamb_reg).onConflict().merge();
                            }
                        }
                    }
                }));
            }
            catch (error) {
                return this.catchError(error);
            }
        });
        this.changeAspirantStatus = (status, jamb_regs) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                yield _1.default.table(db_types_1.ETables.ASPIRANT).update({ status }).whereIn("jamb_reg", jamb_regs).onConflict().merge();
            }
            catch (error) {
                return this.catchError(error);
            }
        });
        this.DBGetAcceptedAspirants = () => tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                const aspirants = yield _1.default.select('*').from(db_types_1.ETables.ASPIRANT).where('status', 'accepted');
                return aspirants;
            }
            catch (error) {
                return this.catchError(error);
            }
        });
        this.DBGetRejectedAspirants = () => tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                const aspirants = yield _1.default.select('*').from(db_types_1.ETables.ASPIRANT).where('status', 'rejected');
                return aspirants;
            }
            catch (error) {
                return this.catchError(error);
            }
        });
        this.DBGetStudentsByProgramme = (programme_id) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                const students = yield _1.default.select([
                    "enrl.*",
                    "std.first_name as first_name",
                    "std.last_name as last_name",
                    "std.email as email",
                    "std.adm_no as adm_no",
                    "std.phone as phone",
                    "std.DOB as DOB",
                    "std.jamb_reg as jamb_reg",
                    "prg.*"
                ])
                    .from({ enrl: db_types_1.ETables.ENROLLMENT })
                    .join({ std: db_types_1.ETables.STUDENT }, "std.id", "enrl.student_id")
                    .leftJoin({ prg: db_types_1.ETables.PROGRAMME }, "prg.id", "enrl.programme_id")
                    .where('programme_id', programme_id);
                return students;
            }
            catch (error) {
                return this.catchError(error);
            }
        });
    }
};
AdminAdapter = tslib_1.__decorate([
    (0, typedi_1.Service)()
], AdminAdapter);
let AdminValidatorAdapter = class AdminValidatorAdapter extends _1.BaseAdapter {
    constructor() {
        super(...arguments);
        this.DBGetUserAndPassword = (email) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                const admin = yield _1.default.select([
                    "adm.email as email",
                    "adm.id as id",
                    "adm.password as password",
                    'role.title as role',
                    'role.permissions as permissions'
                ])
                    .from({ adm: db_types_1.ETables.ADMIN })
                    .join({ role: db_types_1.ETables.ADMIN_ROLE }, 'role.id', 'adm.role_id')
                    .where("email", email).first();
                return admin;
            }
            catch (error) {
                return this.catchError(error);
            }
        });
        this.IsAValidSession = (id) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                const session = yield _1.default.select('*').from(db_types_1.ETables.ACADEMIC_SESSION).where("id", id).first();
                return session;
            }
            catch (error) {
                return this.catchError(error);
            }
        });
        this.DoesProgrammeExist = (id) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                const programme = yield _1.default.select('*').from(db_types_1.ETables.PROGRAMME).where("id", id).first();
                return programme;
            }
            catch (error) {
                return this.catchError(error);
            }
        });
        this.CheckProgrammeByStream = (stream) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                const programme = yield _1.default.select('*').from(db_types_1.ETables.PROGRAMME).where("stream", stream).first();
                return programme;
            }
            catch (error) {
                return this.catchError(error);
            }
        });
        this.DBGetAdmin = (id) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                const admin = yield _1.default.select('*').from(db_types_1.ETables.ADMIN).where("id", id).first();
                return admin;
            }
            catch (error) {
                return this.catchError(error);
            }
        });
        this.checkForValidRole = (role_id) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                const role = yield _1.default.select('*').from(db_types_1.ETables.ADMIN_ROLE).where("id", role_id).first();
                return role;
            }
            catch (error) {
                return this.catchError(error);
            }
        });
        this.checkProgramByCategory = (prog_level) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                const programme = yield _1.default.select('*').from(db_types_1.ETables.PROGRAMME).where("prog_level", prog_level).first();
                return programme;
            }
            catch (error) {
                return this.catchError(error);
            }
        });
        this.DBCheckApplication = (id, stream) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                let application;
                // if (stream === Stream.STANDARD) {
                if (Object.values(enum_1.NID_Stream).includes(stream)) {
                    application = yield _1.default.select('*').from(db_types_1.ETables.ASPIRANT).where("id", id).first();
                }
                else {
                    application = yield _1.default.select('*').from(db_types_1.ETables.PROF_ASPIRANT).where("id", id).first();
                }
                return application;
            }
            catch (error) {
                return this.catchError(error);
            }
        });
    }
};
exports.AdminValidatorAdapter = AdminValidatorAdapter;
exports.AdminValidatorAdapter = AdminValidatorAdapter = tslib_1.__decorate([
    (0, typedi_1.Service)()
], AdminValidatorAdapter);
exports.default = AdminAdapter;
