import { Service } from "typedi";
import BaseController from "..";
import { Request, Response } from "express";
// import { nanoid } from "nanoid";
import * as bcrypt from "bcryptjs"
import AdminAdapter from "../../Database/adapters/adminAdapter";
import { created, success, successAction, successPaginated } from "../../Utils/api_response";
import { PROGRAMMES, Permisions, Status, Stream, permission_divider } from "../../types/enum";
import { generateAdminToken } from "../../Utils/jwt";
import CourseAdapter from "../../Database/adapters/courseAdapter";
import { excelToJson } from "../../Utils/exceljs";
import { sendMail } from "../../Utils/mailer/nodemailer";
import path from "path";

@Service()
class AdminController extends BaseController {
    constructor(
        private readonly adminAdapter: AdminAdapter,
        private readonly courseAdapter: CourseAdapter
    ) {
        super()
    }

    public login = async (req: Request, res: Response) => {
        const { email, id, role } = req.body

        try {
            const token = generateAdminToken({ email, id, role });

            return success(res, { token }, "Logged in successfully");

            // const me = await this.adminAdapter.DBGetMe(id);

            // return success(res, { token, me }, "Logged in successfully");
        } catch (error) {
            return this.catchError(error, res)
        }
    }

    public getDashboardData = async (req: Request, res: Response) => {
        try {
            const data = await this.adminAdapter.DBGetDashboardData();

            return success(res, data);
        } catch (error) {
            return this.catchError(error, res)
        }
    }

    public createAdmin = async (req: Request, res: Response) => {
        const { email, name, password, role_id } = req.body

        try {
            const raw = password;

            const hash = bcrypt.hashSync(raw, 10);

            await this.adminAdapter.DBCreateAdmin({ email, name, password: hash, role_id });

            sendMail({
                to: email,
                type: 'html',
                subject: 'Account credentials',
                content: `
                    <div>
                        <p> Use the following credentials to access the dashboard </p>
                        <p> email: ${email} </p>
                        <p> password: ${raw} </p>
                    </div>
                `
            });

            return created(res, null);
        } catch (error) {
            return this.catchError(error, res)
        }
    }

    public getAdmins = async (req: Request, res: Response) => {
        try {
            const admins = await this.adminAdapter.DBGetAdmins();

            return success(res, admins);
        } catch (error) {
            return this.catchError(error, res)
        }
    }

    public editAdmin = async (req: Request, res: Response) => {
        try {
            const { name, role_id, email } = req.body;
            const { id } = req.params;

            const admins = await this.adminAdapter.DBEditAdmin(+id, { name, role_id, email });

            return success(res, admins);
        } catch (error) {
            return this.catchError(error, res)
        }
    }

    public createAdminRole = async (req: Request, res: Response) => {
        const { title, description, permissions: raw_perms } = req.body as { title: string, description: string, permissions: Permisions[] };

        try {
            const permissions = raw_perms.reduce((acc: string, cur: string) => !acc ? cur : acc + permission_divider + cur, '')

            const role = await this.adminAdapter.DBCreateAdminRole({ title, description, permissions });

            return created(res, role)
        } catch (error) {
            return this.catchError(error, res)
        }
    }

    public getRoles = async (req: Request, res: Response) => {
        try {
            const roles = await this.adminAdapter.DBGetRoles();

            return success(res, roles);
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    public getMe = async (req: Request, res: Response) => {
        try {
            const { auth_user } = req.body
            const me = await this.adminAdapter.DBGetMe(auth_user.id);

            return success(res, me);
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    public createCourse = async (req: Request, res: Response) => {
        try {
            const { code, title, description, unit, type, prog_level } = req.body;

            // const courses = await this.courseAdapter.createCourseV2({ code, title, unit, type, prog_level, description });

            return success(res, 'courses');
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    public getAllSessions = async (req: Request, res: Response) => {
        try {
            const sessions = await this.adminAdapter.DBGetSessions()

            return success(res, sessions);
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    public getAllProgrammes = async (req: Request, res: Response) => {
        try {
            const programmes = await this.adminAdapter.DBGetAllProgrammes()

            return success(res, programmes);
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    public getProgrammeByCategory = async (req: Request, res: Response) => {
        try {
            const { stream } = req.params as { stream: Stream };
            const programme = await this.adminAdapter.DBGetProgramme(stream);

            return success(res, programme);
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    public getStudents = async (req: Request, res: Response) => {
        try {
            const students = await this.adminAdapter.DBGetStudents();

            return success(res, students);
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    public getCourses = async (req: Request, res: Response) => {
        try {
            const { stream } = req.params as { stream: Stream }

            const courses = await this.adminAdapter.DBGetCourses(stream);

            return success(res, courses);
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    public getACourse = async (req: Request, res: Response) => {
        try {
            const { stream, id } = req.params as unknown as { stream: Stream, id: number }

            const course = await this.adminAdapter.DBGetACourse(stream, id);

            return success(res, course);
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    public getAssignedCourses = async (req: Request, res: Response) => {
        try {
            const { auth_user } = req.body;
            const { stream } = req.params as { stream: Stream }

            const assigned_courses = await this.adminAdapter.DBGetAssignedCourses(auth_user.id, stream);

            return success(res, assigned_courses);
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    public assignStaff = async (req: Request, res: Response) => {
        try {
            const { stream, id } = req.params as unknown as { stream: Stream, id: number };
            const { admin_ids, session } = req.body;

            await this.adminAdapter.DBAssignStaff(stream, id, admin_ids, session.id);

            return successAction(res);
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    public getAssignedStaff = async (req: Request, res: Response) => {
        try {
            const { stream, id } = req.params as unknown as { stream: Stream, id: number };
            const { session } = req.body;

            const staff = await this.adminAdapter.DBGetAssignedStaff(stream, id, session.id);

            return success(res, staff);
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    public getPendingApplications = async (req: Request, res: Response) => {
        try {
            const { stream } = req.params as { stream: Stream }

            const applicatiions = await this.adminAdapter.DBGetPendingApplications(stream);

            return success(res, applicatiions)
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    public getApplication = async (req: Request, res: Response) => {
        try {
            const { stream, id } = req.params as unknown as { stream: Stream, id: number }

            const application = await this.adminAdapter.DBGetApplication(id, stream);

            return success(res, application)
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    public reviewApplication = async (req: Request, res: Response) => {
        try {
            const { id, stream } = req.params as unknown as { id: number, stream: Stream };
            const { status, comment, programme_id } = req.body;

            // if (stream === Stream.STANDARD) {
            //     await this.adminAdapter.DBReviewApplication(+id, stream, { programme_id, status });
            // } else {
            //     await this.adminAdapter.DBReviewApplication(+id, stream, { programme_id, status });
            // }
            await this.adminAdapter.DBReviewApplication(+id, stream, { programme_id, status });

            // send comment along with noftification via mail
            return successAction(res);
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    public getCourseResult = async (req: Request, res: Response) => {
        try {
            const { id, session_id } = req.params as unknown as { id: number, session_id: number }

            const result = await this.adminAdapter.DBGetCourseResult(id, session_id);

            return success(res, result);
        } catch (error) {
            return this.catchError(error, res);
        }
    }
















    // PROFESSIONAL





    // NID

    // STUDENT

    public uploadResult = async (req: Request, res: Response) => {
        try {
            const { course_id, level, session, sheetName } = req.body;
            const { stream } = req.params;
            const { buffer, mimetype } = req.file!;

            const base64String = buffer.toString('base64');
            console.log(course_id, 'course_id')
            const results = excelToJson(buffer, sheetName);
            console.log(results, 'RESLST')
            // console.log(results, 'results');

            // const course = await this.courseAdapter.DBGetCourseById(+course_id);

            const getMatricNos = results.map(res => {
                const matricKey = Object.keys(res).find(key => key.toLowerCase().includes('matric'));

                return matricKey ? res[matricKey] : undefined
            }).filter(m => m !== undefined);
            console.log(getMatricNos, 'getMatricNos')
            const students = await this.adminAdapter.DBGetSelectedStudents(getMatricNos);
            console.log(students, 'STD')
            const updates = [];
            for (let student of students) {
                let std_result;
                for (let result of results) {
                    const matric_key = Object.keys(result).find(key => key.toLowerCase().includes('matric'))!
                    const CA_key = Object.keys(result).find(key => key.toLowerCase().includes('ca'))!
                    const exam_key = Object.keys(result).find(key => key.toLowerCase().includes('exam'))!
                    console.log(result[matric_key], 'key')
                    console.log(student.matric_no, 'mat')
                    if (result[matric_key] === student.matric_no) {
                        std_result = result;

                        const obj = {
                            student_id: student.id, CA: result[CA_key], exam: result[exam_key], grade: (result[CA_key] + result[exam_key])
                        }
                        updates.push(obj);
                    }
                }
            }

            // update result for the concerned students
            updates.length && await this.adminAdapter.DBUploadResultV2(updates, course_id, session.id, level);

            return successAction(res);
        } catch (error) {
            return this.catchError(error, res);
        }
    }





























    public getAllFee = async (req: Request, res: Response) => {
        try {
            console.log('getting')
            const fees = await this.adminAdapter.DBGetAllFee()

            return success(res, fees);
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    public getOtherFee = async (req: Request, res: Response) => {
        try {
            console.log('getting')
            const fees = await this.adminAdapter.DBGetOtherFee()

            return success(res, fees);
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    public setFee = async (req: Request, res: Response) => {
        try {
            const { name, amount, prog_level, programme_id, session, category, is_recurrent, session_id } = req.body;
            await this.adminAdapter.DBSetFee({ name, amount, prog_level, programme_id, category, is_recurrent, session_id })

            return successAction(res);
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    public uploadAspirants = async (req: Request, res: Response) => {
        try {
            const { session_id, year, sheetName } = req.body;
            const { buffer, mimetype } = req.file!;

            const base64String = buffer.toString('base64');
            // await this.adminAdapter.DBUploadPutmeResult(type, session, base64String, year);

            const aspirants = excelToJson(buffer, sheetName);

            await this.adminAdapter.DBUploadAspirants({ session_id: +session_id, file: buffer, mimetype, sheetName, year }, aspirants);

            return successAction(res);
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    public getAspirants = async (req: Request, res: Response) => {
        try {
            const aspirants = await this.adminAdapter.DBGetAspirants();

            return success(res, aspirants)
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    public viewAspirant = async (req: Request, res: Response) => {
        try {
            const { jamb_reg } = req.params;

            const aspirants = await this.adminAdapter.DBViewAspirant(jamb_reg);

            return success(res, aspirants)
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    public uploadPutmeResults = async (req: Request, res: Response) => {
        try {
            const { type, session_id, year, sheetName } = req.body;
            const { buffer, mimetype } = req.file!;

            const base64String = buffer.toString('base64');
            // await this.adminAdapter.DBUploadPutmeResult(type, session, base64String, year);

            const results = excelToJson(buffer, sheetName);

            const aspirants = await this.adminAdapter.DBGetAspirants();

            await this.adminAdapter.DBUploadPutmeResult({ type, session_id: +session_id, file: buffer, mimetype, sheetName, year }, aspirants, results);

            return successAction(res);
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    public changeAspirantStatus = async (req: Request, res: Response) => {
        try {
            const { jamb_regs } = req.body;
            const { status } = req.query as { status: Status }

            await this.adminAdapter.changeAspirantStatus(status, jamb_regs);

            return successAction(res)
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    public getAcceptedAspirants = async (req: Request, res: Response) => {
        try {
            const aspirants = await this.adminAdapter.DBGetAcceptedAspirants();

            return success(res, aspirants);
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    public getRejectedAspirants = async (req: Request, res: Response) => {
        try {
            const aspirants = await this.adminAdapter.DBGetRejectedAspirants();

            return success(res, aspirants);
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    public getStudentsByProgramme = async (req: Request, res: Response) => {
        try {
            const { programme_id } = req.params as any;
            const students = await this.adminAdapter.DBGetStudentsByProgramme(+programme_id);

            return success(res, students);
        } catch (error) {

        }
    }

    public download = async (req: Request, res: Response) => {
        const { file_path } = req.params;
        const decodedFilePath = decodeURIComponent(file_path);
        const filePath = path.join(__dirname, '..', '..', '..', decodedFilePath);
  
        res.sendFile(filePath, err => {
            if (err) {
                console.error('Error sending file:', err);
                res.status(500).send('Error downloading file');
            }
        });
    }

    public testFileUpload = async (req: Request, res: Response) => {
        try {
            console.log(req.body)
            console.log(req.file, 'file')
            // console.log(req, 'req')
        } catch (error) {
            return this.catchError(error, res);
        }
    }

}

export default AdminController;
