import { Inject, Service } from "typedi";
import IndexController from "..";
import { Request, Response } from "express";
import AspirantAdapter from "../../Database/adapters/aspirantAdapter";
import { generateAspirantToken, generateProfAspirantFormToken, generateProfAspirantToken } from "../../Utils/jwt";
import axios from 'axios';
// import { encodeOTP, generateToken } from "../../Utils/jwt";
import { created, success, successAction } from "../../Utils/api_response";
import { calculateCharges, encodeLastName, fileStore, generateApplicationNo, generateRegNo, generateTransactionID, pinGenerator } from "../../Utils/factory";
import { generateContent } from "../../Utils/mailer";
import { sendMail } from "../../Utils/mailer/nodemailer";
import FeeAdapter from "../../Database/adapters/feeAdapter";
import { Invoice_Type, PROF_Stream, Stream } from "../../types/enum";
import Paystack from "../../Utils/paystack";
import { generateApplicationForm } from "../../Utils/pdf/jspdf";
import path from "path";
import { deleteFiles } from "../../Utils/multer";
// import { cleanupTempFiles, moveFilesToUserFolder } from "../../Utils/multer";
// import { cleanupTempFiles, moveFilesOutOfTempFolder } from "../../Utils/multer/index old_v2";

@Service()
class AspirantController extends IndexController {
    constructor(
        @Inject() private readonly aspirantAdapter: AspirantAdapter,
        private readonly feeAdapter: FeeAdapter,
        private readonly paystack: Paystack
    ) {
        super()
    }

    public test = async (req: Request, res: Response) => {
        try {
            const fees = await this.feeAdapter.DBTest();

            return success(res, fees);
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    public getProfProgrammes = async (req: Request, res: Response) => {
        try {
            const programmes = await this.aspirantAdapter.DBGetProfProgrammes();

            return success(res, programmes);
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    public getProfile = async (req: Request, res: Response) => {
        try {
            const { auth_user } = req.body;

            return success(res, { ...auth_user });
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    public signup = async (req: Request, res: Response) => {
        try {
            const { email, first_name, last_name, middle_name, phone, stream } = req.body;

            // const salt = await bcrypt.genSalt(10);
            // const password = bcrypt.hashSync(raw, salt);
            // const pin = pinGenerator(4);
            // const verification_code = encodeOTP(pin, email);
            const token = generateProfAspirantFormToken({ email, first_name, last_name, stream, middle_name, phone })
            const verification_link = `${process.env.TNM_CLIENT_URL}/aspirant/professional/fee/application_fee?token=${token}`;
            // const verification_link = `${process.env.TNM_CLIENT_URL}/auth/verify?code=${pin}&email=${email}&stream=${stream}`;

            const replacements = {
                studentName: `${first_name} ${last_name}`,
                companyName: process.env.COMPANY_NAME!,
                verificationLink: verification_link,
                supportLink: process.env.SUPPORT_LINK!
            };

            // Get email content with dynamic data
            const content = generateContent('/mails/email_verification.html', replacements);
            sendMail({ to: email, type: 'html', subject: 'Please verify your email', content });

            console.log(`verification mail sent to ${email}`);
            console.log(verification_link)

            return successAction(res, "Check your the email provided to continue");
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    public verifyEmail = async (req: Request, res: Response) => {
        const { aspirant, session } = req.body;
        const { email, first_name, last_name, middle_name, stream } = aspirant

        try {
            const fee = await this.feeAdapter.DBGetProfFormFee(session.id);
            fee.charges = calculateCharges(fee.amount);

            return success(res, { fee, aspirant: { first_name, last_name, middle_name, email, stream } });
        } catch (error) {
            return this.catchError(error, res)
        }
    }

    public login = async (req: Request, res: Response) => {
        try {
            const { email, session } = req.body;

            const aspirant = await this.aspirantAdapter.DBGetProfAspirant(email);

            const token = generateProfAspirantToken({ id: aspirant.id, email, stream: aspirant.programme_stream, status: aspirant.status });

            return success(res, { token, aspirant, session }, "Logged in successfully");
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    public getApplication = async (req: Request, res: Response) => {
        try {
            const { email, session } = req.body;

            const aspirant = await this.aspirantAdapter.DBGetProfAspirant(email);

            // const token = generateProfAspirantToken({ id: aspirant.id, email, stream: aspirant.programme_stream, status: aspirant.status });

            // return success(res, { token, aspirant, session }, "Logged in successfully");
            return success(res, aspirant);
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    public apply = async (req: Request, res: Response) => {
        try {
            const { auth_user, middle_name, DOB, gender, qualification, years_of_exp, workplace, session } = req.body;
            const { email, first_name, last_name } = auth_user;
            const { stream } = req.params as { stream: PROF_Stream };

            await this.aspirantAdapter.DBCreateApplication({ first_name, last_name, middle_name, DOB, gender, qualification, years_of_exp, workplace }, email, stream)

            await sendMail({ to: email, type: 'html', subject: 'TNM Application', content: `Hello ${last_name} ${first_name}, Your application is under review, kindly check back in at most 24hrs later. Thanks for your interest` });
            console.log('sending mail to them tellig them their applicationn is under review and should check back in 24hrs', 'so they should be on the lookout, it gona be through the email they supplied');

            sendMail({ to: process.env.ADMISSION_EMAIL || 'hoismail1430@gmail.com', type: 'html', subject: 'One new application at TNM', content: `Hello, there is a new applicationn that's need to be responded to ASAP. Clickk on the link below to view\n ...... ` });
            console.log('sending maill to the admin too, notifying then of a new application')

            return successAction(res, "Application successful");
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    // NID

    public register = async (req: Request, res: Response) => {
        try {
            const { email, first_name, last_name, middle_name, DOB, phone, gender, jamb_reg, session } = req.body;
            // const { o_level, jamb_result } = req.files;
            const files = req.files as { [fieldname: string]: Express.Multer.File[] };
            const { o_level, jamb_result } = files;
            console.log(req.files)
            console.log(req.body, 'body in contrl')
            const password = await encodeLastName(last_name)

            // let receipt: { [key: string]: object } = {};

            const programme = await this.aspirantAdapter.DBGetProgrammeByStream(Stream.FULL);
            const fee = await this.feeAdapter.DBGetPUTMFee(session.id);

            let amount = fee.amount + calculateCharges(fee.amount);
            amount *= 100;

            const transactionID = generateTransactionID();

            fee[transactionID] = transactionID;

            // fileStore.set(email, {
            //     o_level: o_level[0].buffer.toString('base64'),
            //     jamb_result: jamb_result[0].buffer.toString('base64')
            // });
            const olevel_file_path = path.join(process.env.UPLOAD_PATH ?? 'public/uploads', o_level[0].filename); // path relative to public_html
            const jamb_result_file_path = path.join(process.env.UPLOAD_PATH ?? 'public/uploads', jamb_result[0].filename);

            const metadata = {
                fee,
                customer: {
                    first_name, last_name, middle_name, email, phone, DOB, gender, jamb_reg,
                    olevel_file_path, jamb_result_file_path,
                    o_level_mimetype: o_level[0].mimetype,
                    jamb_result_mimetype: jamb_result[0].mimetype
                },
                programme_id: programme.id,
                stream: Stream.FULL,
                invoice_type: Invoice_Type.PUTME
            };

            const payment = await this.paystack.initializeTransaction({ email, amount: Math.ceil(amount), reference: transactionID, metadata, callback_url: `${process.env.TNM_CLIENT_URL}/aspirant/nid/login` })

            // return success(res, { url: payment.authorization_url, fee_id: fee.id });
            return success(res, { url: '#', fee_id: fee.id });
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    public nidLogin = async (req: Request, res: Response) => {
        try {
            const { jamb_reg, session } = req.body;

            const aspirant = await this.aspirantAdapter.DBGetAspirant(jamb_reg);

            const token = generateAspirantToken({ id: aspirant.id, email: aspirant.email, stream: aspirant.programme_stream, jamb_reg });

            return success(res, { token }, "Logged in successfully");
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    public uploadAdmissiionLetter = async (req: Request, res: Response) => {
        try {
            const { auth_user } = req.body;
            // const { buffer, mimetype } = req.file!;
            const file = req.file!;

            // await this.aspirantAdapter.DBUploadAdmissionLetter(auth_user.jamb_reg, { adm_letter: buffer.toString('base64'), adm_letter_mimetype: mimetype });
            // await this.aspirantAdapter.DBUploadAdmissionLetter(auth_user.jamb_reg, { adm_letter: buffer, adm_letter_mimetype: mimetype });
            await this.aspirantAdapter.DBUploadAdmissionLetter(auth_user.jamb_reg, { adm_letter_path: file.path, adm_letter_mimetype: file.mimetype });

            return successAction(res);
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    // NID Parrt Time
    public partTimeApplication = async (req: Request, res: Response) => {
        try {
            const { email, first_name, last_name, middle_name, DOB, phone, gender, state_of_origin, lga, address, manual_o_level, session } = req.body;
            // const { o_level, jamb_result } = req.files;
            const files = req.files as { [fieldname: string]: Express.Multer.File[] };
            const { o_level, passport } = files;
            console.log(req.files)
            console.log(req.body, 'body in contrl')
            const password = await encodeLastName(last_name)

            // let receipt: { [key: string]: object } = {};

            const programme = await this.aspirantAdapter.DBGetProgrammeByStream(Stream.PART);
            const fee = await this.feeAdapter.DBGetPartApplicationFee(session.id);

            let amount = fee.amount + calculateCharges(fee.amount);
            amount *= 100;

            const transactionID = generateTransactionID();

            fee[transactionID] = transactionID;

            // const olevel_file_path = path.join(process.env.UPLOAD_PATH ?? 'public/uploads', o_level[0].filename); // path relative to public_html
            // const passport_file_path = path.join(process.env.UPLOAD_PATH ?? 'public/uploads', passport[0].filename);
            const olevel_file_path = o_level[0].path;
            const passport_file_path = passport[0].path;

            const reg_no = await generateApplicationNo();

            const new_aspirant = await this.aspirantAdapter.DBPartApplication({
                first_name, last_name, middle_name, email, phone, password, DOB, gender, state_of_origin, lga, address, manual_o_level,
                o_level_path: olevel_file_path, o_level_mimetype: o_level[0].mimetype,
                passport_path: passport_file_path, passport_mimetype: passport[0].mimetype

            }, reg_no, programme.id, session);

            // await moveFilesOutOfTempFolder(req.body.tempUploadDir);

            sendMail({ to: email, type: 'html', subject: 'PUTME Registration', content: `<div> Thanks for your interest in TNM Media Academy, kindly login to continue using your email and surname as the password </div> ` });

            return successAction(res, 'Application completed successfully. Kindly login to generate your application letter');
            // const metadata = {
            //     fee,
            //     customer: {
            //         first_name, last_name, middle_name, email, phone, DOB, gender, state_of_origin, lga, address,
            //         manual_o_level,
            //         olevel_file_path, passport_file_path,
            //         o_level_mimetype: o_level[0].mimetype, passport_mimetype: passport[0].mimetype
            //     },
            //     programme_id: programme.id,
            //     stream: Stream.PART,
            //     invoice_type: Invoice_Type.PART_FORM
            // };

            // const payment = await this.paystack.initializeTransaction({ email, amount: Math.ceil(amount), reference: transactionID, metadata, callback_url: `${process.env.TNM_CLIENT_URL}/aspirant/nid/login` })

            // return success(res, { url: payment.authorization_url, fee_id: fee.id });
            // return success(res, { url: '#', fee_id: fee.id });
        } catch (error) {
            // await cleanupTempFiles(req.body.tempUploadDir);
            if (req.body.uploadedFiles && req.body.uploadedFiles.length > 0) {
                deleteFiles(req.body.uploadedFiles);
            }
            this.catchError(error, res);
        }
    }

    public partLogin = async (req: Request, res: Response) => {
        try {
            const { email, session } = req.body;

            const aspirant = await this.aspirantAdapter.DBGetAspirantByEmail(email);

            const token = generateAspirantToken({ id: aspirant.id, email: aspirant.email, stream: aspirant.programme_stream });

            return success(res, { token }, "Logged in successfully");
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    public generateApplicationForm = async (req: Request, res: Response) => {
        try {
            const { auth_user } = req.body
            const output = await generateApplicationForm(auth_user);

            // Set the content type to application/pdf
            res.setHeader('Content-Type', 'application/pdf');
            // Suggest download file name
            res.setHeader('Content-Disposition', `attachment; filename=application-form-${auth_user.first_name.replace(/\s+/g, '-')}.pdf`);

            // Send the PDF as a buffer
            res.send(Buffer.from(output));
        } catch (error) {
            return this.catchError(error, res);
        }
    }

};

export default AspirantController;