import { Inject, Service } from "typedi";
import IndexController from "..";
import { Request, Response } from "express";
import axios from 'axios';
import bcrypt from 'bcryptjs';
import forge from 'node-forge'
import { badRequest, success, successAction } from "../../Utils/api_response";
import FeeAdapter from "../../Database/adapters/feeAdapter";
import { calculateCharges, encodeLastName, fileStore, generateApplicationNo, generateRegNo, generateTransactionID } from "../../Utils/factory";
import Pay_refAdapter from "../../Database/adapters/pay_refAdapter";
import { generateContent } from "../../Utils/mailer";
import { sendMail } from "../../Utils/mailer/nodemailer";
import StudentAdapter from "../../Database/adapters/studentAdapter";
import AspirantAdapter from "../../Database/adapters/aspirantAdapter";
import { Category, Invoice_Type, Level, NID_Stream, PROF_Stream, Stream } from "../../types/enum";

@Service()
class FeeController extends IndexController {
    constructor(
        private readonly feeAdapter: FeeAdapter,
        private readonly pay_refAdapter: Pay_refAdapter,
        private readonly studentAdapter: StudentAdapter,
        private readonly aspirantAdapter: AspirantAdapter
    ) {
        super()
    }

    public payProfFormFee = async (req: Request, res: Response) => {
        try {
            const { first_name, last_name, middle_name, email, phone, session } = req.body;
            const { stream } = req.params;
            const { split } = req.query as unknown as { split: boolean }

            let fee = await this.feeAdapter.DBGetProfFormFee(session.id);
            let amount = 0
            // const totalFee = fee.amount + calculateCharges(fee.amount);
            if (split && fee.upfront) {
                // calculate 60% of it
                // amount = 0.6 * totalFee * 100;
                amount = ((0.6 * fee.amount ) + calculateCharges(fee.amount)) * 100;
                fee = {
                    ...fee,
                    amount_paid: amount,
                    amount_due: fee.amount,
                    balance: fee.amount - amount
                }
            } else {
                // amount = totalFee * 100;
                amount = (fee.amount + calculateCharges(fee.amount)) * 100;
            }
            const transactionID = generateTransactionID();

            const metadata = {
                fee,
                customer: { first_name, last_name, middle_name, email, phone },
                stream,
                invoice_type: Invoice_Type.FORM
            };

            const payment = await axios({
                method: 'POST',
                url: `https://api.paystack.co/transaction/initialize`,
                headers: {
                    "Content-Type": "application/json",
                    Authorization: `Bearer ${process.env.PAYSTACK_SECRET_KEY}`
                },
                // data: { email, amount: Math.ceil(amount), reference: transactionID, metadata, callback_url: `${process.env.TNM_CLIENT_URL}/aspirant/professional/fee/application_fee/verify?fee_id=${fee.id}&stream=${stream}` },
                data: { email, amount: Math.ceil(amount), reference: transactionID, metadata, callback_url: `${process.env.TNM_CLIENT_URL}/aspirant/professional/login?msg=Kindly fllow the instructions you received in teh mail to continue` },
            })

            // return success(res, { url: payment.data.data.authorization_url, ref: transactionID, fee_id: fee.id });
            return success(res, { url: '#', ref: transactionID, fee_id: fee.id });
        } catch (error) {
            return this.catchError(error, res)
        }
    }

    public processFormPayment = async (session: any, amount: number, metadata: any, reference: string) => {
        try {
            const { email, first_name, last_name, middle_name, DOB, phone } = metadata.customer

            console.log(last_name.toLowerCase(), 'paswoord forlogin')
            const salt = await bcrypt.genSalt(10);
            const password = bcrypt.hashSync(last_name.toLowerCase(), salt);

            const new_aspirant = await this.aspirantAdapter.DBCreateProfAspirant({ email, first_name, last_name, password, middle_name, phone }, metadata.stream, session.id, session.current_diet.id);
            console.log(new_aspirant, 'new aspirant');
            await this.feeAdapter.DBCreateAspirantFee(new_aspirant, metadata.fee, amount / 100, metadata.stream, reference, session);

            sendMail({ to: email, type: 'html', subject: 'Apllication Request', content: `<div> Kindly login to continue, use your email as the username and surname as the password </div>` });
        } catch (error) {
            throw error;
        }
    }

    public verifyFormPayment = async (req: Request, res: Response) => {
        try {
            // const { email, first_name, last_name, DOB, session } = req.body;
            const { session, fee_id } = req.body;
            const { stream } = req.params as any
            const { reference } = req.query;

            const verify = await axios({
                method: 'GET',
                url: `https://api.paystack.co/transaction/verify/${reference}`,
                headers: {
                    "Content-Type": "application/json",
                    Authorization: `Bearer ${process.env.PAYSTACK_SECRET_KEY}`
                },
            });
            const { status, amount, gateway_response, requested_amount, channel, metadata, message } = verify.data.data
            console.log(metadata, 'metadata')
            console.log(status, 'status')
            if (status !== 'success') return badRequest(res, gateway_response)

            if (amount !== requested_amount) throw new Error('Invalid payment');

            const { email, first_name, last_name, middle_name, DOB, phone } = metadata.customer

            const ref = reference as string
            await this.pay_refAdapter.DBSavePaymentRef(email, ref, metadata.fee);

            const replacements = {
                userName: `${first_name} ${last_name}`,
                companyName: process.env.COMPANY_NAME || 'TNM Media Academy',
                orderID: ref,
                orderDate: new Date(),
                paymentMethod: channel,
                shippingAddress: "",
                deliveryDate: "3 days",
                contactInfo: 'tnm_contact',
                supportLink: process.env.SUPPORT_LINK!
            };

            // Get email content with dynamic data
            const emailContent = generateContent('/mails/receipt.html', replacements);
            sendMail({ to: email, type: 'html', subject: 'Please verify your email', content: emailContent });

            // crete sudent_fee data
            if (status) {
                console.log(last_name.toLowerCase(), 'paswoord forlogin')
                const salt = await bcrypt.genSalt(10);
                const password = bcrypt.hashSync(last_name.toLowerCase(), salt);

                const new_aspirant = await this.aspirantAdapter.DBCreateProfAspirant({ email, first_name, last_name, password, middle_name, phone }, stream, session.id, session.current_diet.id);
                console.log(new_aspirant, 'new aspirant');
                await this.feeAdapter.DBCreateAspirantFee(new_aspirant, metadata.fee, amount / 100, stream, ref, session);
            }

            return success(res, status, 'Kindly login to continue, use your email as the username and surname as the password');
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    // NID

    public getPUTMEFee = async (req: Request, res: Response) => {
        try {
            const { session } = req.body;

            const fee = await this.feeAdapter.DBGetPUTMFee(session.id);
            fee.charges = calculateCharges(fee.amount)
            // return success(res, {fee, charges: calculateCharges(fee.amount)});
            return success(res, fee);
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    public processPUTMEPayment = async (session: any, amount: number, metadata: any, reference: string) => {
        try {
            const { email, first_name, last_name, middle_name, DOB, phone, gender, jamb_reg, o_level_mimetype, jamb_result_mimetype, olevel_file_path, jamb_result_file_path } = metadata.customer

            // const fileData = fileStore.get(email);

            const salt = await bcrypt.genSalt(10);
            const password = bcrypt.hashSync(last_name.toLowerCase(), salt);
            const reg_no = await generateApplicationNo();

            const new_aspirant = await this.aspirantAdapter.DBPUTMERegistration({
                first_name, last_name, middle_name, email, phone, password, DOB, gender, jamb_reg,
                // o_level: Buffer.from(fileData.o_level, 'base64'), o_level_mimetype,
                // jamb_result: Buffer.from(fileData.jamb_result, 'base64'), jamb_result_mimetype

                // o_level: fileData.o_level, o_level_mimetype,
                // jamb_result: fileData.jamb_result, jamb_result_mimetype
                o_level_path: olevel_file_path, o_level_mimetype,
                jamb_result_path: jamb_result_file_path, jamb_result_mimetype
            }, reg_no, +metadata.programme_id, session)

            console.log(new_aspirant, 'new aspirant');
            await this.feeAdapter.DBCreateAspirantFee(new_aspirant, metadata.fee, amount / 100, metadata.stream, reference, session);

            fileStore.delete(email);

            sendMail({ to: email, type: 'html', subject: 'PUTME Registration', content: `<div> Thanks for your interest in TNM Media Academy, kindly login to continue using your jamb number and surname as the password </div> ` });
        } catch (error) {
            throw error;
        }
    }

    public verifyPUTMEPayment = async (req: Request, res: Response) => {
        try {
            const { session, fee_id } = req.body;
            const { reference } = req.query;

            const verify = await axios({
                method: 'GET',
                url: `https://api.paystack.co/transaction/verify/${reference}`,
                headers: {
                    "Content-Type": "application/json",
                    Authorization: `Bearer ${process.env.PAYSTACK_SECRET_KEY}`
                },
            });
            const { status, amount, gateway_response, requested_amount, channel, metadata, message } = verify.data.data
            console.log(metadata, 'metadata')
            // const { email, first_name, last_name, middle_name, DOB, phone, gender, jamb_reg, o_level, o_level_mimetype, jamb_result, jamb_result_mimetype } = metadata.customer
            const { email, first_name, last_name, middle_name, DOB, phone, gender, jamb_reg, o_level_mimetype, jamb_result_mimetype } = metadata.customer
            const fileData = fileStore.get(email);

            console.log(status, 'status')
            if (status !== 'success') return badRequest(res, gateway_response)

            if (amount !== requested_amount) throw new Error('Invalid payment');

            const ref = reference as string
            await this.pay_refAdapter.DBSavePaymentRef(email, ref, metadata.fee);

            const replacements = {
                userName: `${first_name} ${last_name}`,
                companyName: process.env.COMPANY_NAME || 'TNM Media Academy',
                orderID: ref,
                orderDate: new Date(),
                paymentMethod: channel,
                shippingAddress: "",
                deliveryDate: "3 days",
                contactInfo: 'tnm_contact',
                supportLink: process.env.SUPPORT_LINK!
            };

            // Get email content with dynamic data
            const emailContent = generateContent('/mails/receipt.html', replacements);
            sendMail({ to: email, type: 'html', subject: 'Please verify your email', content: emailContent });

            // crete sudent_fee data
            if (status) {
                const salt = await bcrypt.genSalt(10);
                const password = bcrypt.hashSync(last_name.toLowerCase(), salt);
                const reg_no = await generateApplicationNo();

                // const new_aspirant = await this.aspirantAdapter.DBPUTMERegistration({
                //     first_name, last_name, middle_name, email, phone, password, DOB, gender, jamb_reg,
                //     // o_level: Buffer.from(fileData.o_level, 'base64'), o_level_mimetype,
                //     // jamb_result: Buffer.from(fileData.jamb_result, 'base64'), jamb_result_mimetype
                //     o_level: fileData.o_level, o_level_mimetype,
                //     jamb_result: fileData.jamb_result, jamb_result_mimetype
                // }, reg_no, metadata.programme_id, session)

                // console.log(new_aspirant, 'new aspirant');
                // await this.feeAdapter.DBCreateAspirantFee(new_aspirant, metadata.fee, amount / 100, metadata.stream, ref, session);

                fileStore.delete(email);
            }

            return success(res, status);
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    // PART

    public getPartApplicationFee = async (req: Request, res: Response) => {
        try {
            const { session } = req.body;

            const fee = await this.feeAdapter.DBGetPartApplicationFee(session.id);
            fee.charges = calculateCharges(fee.amount)
            // return success(res, {fee, charges: calculateCharges(fee.amount)});
            return success(res, fee);
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    public processPartApplicationPayment = async (session: any, amount: number, metadata: any, reference: string) => {
        try {
            const { email, first_name, last_name, middle_name, DOB, phone, gender, state_of_origin, lga, address, manual_o_level, o_level_mimetype, passport_mimetype, olevel_file_path, passport_file_path } = metadata.customer

            // const fileData = fileStore.get(email);

            const salt = await bcrypt.genSalt(10);
            const password = bcrypt.hashSync(last_name.toLowerCase(), salt);
            const reg_no = await generateApplicationNo();

            const new_aspirant = await this.aspirantAdapter.DBPartApplication({
                first_name, last_name, middle_name, email, phone, password, DOB, gender, state_of_origin, lga, address, manual_o_level,
                // o_level: fileData.o_level, o_level_mimetype,
                // passport: fileData.passport, passport_mimetype
                o_level_path: olevel_file_path, o_level_mimetype,
                passport_path: passport_file_path, passport_mimetype

            }, reg_no, +metadata.programme_id, session)

            console.log(new_aspirant, 'new aspirant');
            await this.feeAdapter.DBCreateAspirantFee(new_aspirant, metadata.fee, amount / 100, metadata.stream, reference, session);

            fileStore.delete(email);

            sendMail({ to: email, type: 'html', subject: 'PUTME Registration', content: `<div> Thanks for your interest in TNM Media Academy, kindly login to continue using your email and surname as the password </div> ` });
        } catch (error) {
            throw error;
        }
    }

    // BOTH

    public getIDCardFee = async (req: Request, res: Response) => {
        try {
            const { session } = req.body;
            const { auth_user } = req.body;

            // const category = auth_user.stream ? Category.PROFESSIONAL : Category.STANDARD
            const category = Object.values(NID_Stream).includes(auth_user.stream) ? Category.NID : Category.PROFESSIONAL;
            const fee = await this.feeAdapter.DBGetIDCardFee(session.id, category)

            return success(res, fee);
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    public getAcceptanceFee = async (req: Request, res: Response) => {
        try {
            const { session } = req.body;
            const { auth_user } = req.body;

            // const category = auth_user.stream ? Category.PROFESSIONAL : Category.STANDARD
            const category = (auth_user.stream === PROF_Stream.BASIC || auth_user.stream === PROF_Stream.ADVANCED) ? Category.PROFESSIONAL : auth_user.stream === NID_Stream.FULL ?  Category.FULL : Category.PART
            const fee = await this.feeAdapter.DBGetAcceptanceFee(session.id, category);

            return success(res, fee);
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    public payAcceptanceFee = async (req: Request, res: Response) => {
        try {
            const { auth_user, session } = req.body;
            const { email, first_name, last_name } = auth_user;
            const { programme_id, stream } = req.params as any;
            const { split } = req.query as unknown as { split: boolean }

            // const category = auth_user.stream ? Category.PROFESSIONAL : Category.STANDARD
            const category = (auth_user.stream === PROF_Stream.BASIC || auth_user.stream === PROF_Stream.ADVANCED) ? Category.PROFESSIONAL : auth_user.stream === NID_Stream.FULL ?  Category.FULL : Category.PART
            let fee = await this.feeAdapter.DBGetAcceptanceFee(session.id, category);
            // let receipt: { [key: string]: object } = {}; 
            let amount;
            const transactionID = generateTransactionID();

            fee[transactionID] = transactionID;

            if (split && fee.upfront) {
                // calculate 60% of it
                amount = ((0.6 * fee.amount ) + calculateCharges(fee.amount)) * 100;
                fee = {
                    ...fee,
                    amount_paid: amount,
                    amount_due: fee.amount,
                    balance: fee.amount - amount
                }
            } else {
                amount = (fee.amount + calculateCharges(fee.amount)) * 100;
            }

            delete auth_user.o_level
            delete auth_user.jamb_result
            delete auth_user.adm_letter;

            const metadata = {
                fee,
                customer: { first_name, last_name, email },
                stream: auth_user.stream ?? stream,
                invoice_type: Invoice_Type.ACCEPTANCE,
                auth_user
            };

            // const callback_url = stream ? `${process.env.TNM_CLIENT_URL}/aspirant/professional/fee/acceptance_fee/verify?fee_id=${fee.id}&stream=${stream}` :
            //     `${process.env.TNM_CLIENT_URL}/aspirant/nid/fee/acceptance_fee/verify?fee_id=${fee.id}`
            const callback_url = stream ? `${process.env.TNM_CLIENT_URL}/aspirant/professional` :
                `${process.env.TNM_CLIENT_URL}/aspirant/nid`

            const payment = await axios({
                method: 'POST',
                url: `https://api.paystack.co/transaction/initialize`,
                headers: {
                    "Content-Type": "application/json",
                    Authorization: `Bearer ${process.env.PAYSTACK_SECRET_KEY}`
                },
                data: { email, amount: Math.ceil(amount), reference: transactionID, metadata, callback_url },
            })

            console.log(transactionID)

            // return success(res, { url: payment.data.data.authorization_url, ref: transactionID, fee_id: fee.id });
            return success(res, { url: '#', ref: transactionID, fee_id: fee.id });
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    public processAcceptanceFee = async (session: any, amount: number, metadata: any, reference: string) => {
        try {
            const { auth_user, fee, stream } = metadata

            await this.feeAdapter.DBCreateAspirantFee(auth_user.id, fee, amount / 100, stream, reference, session);
            await this.aspirantAdapter.DBUpdateAcceptancePaymentStatus(auth_user.id, auth_user.jamb_reg);
        } catch (error) {
            throw error;
        }
    }

    public verifyAcceptanceFee = async (req: Request, res: Response) => {
        try {
            const { session, auth_user } = req.body;
            const { stream } = req.params as any;
            const { reference } = req.query;

            const verify = await axios({
                method: 'GET',
                url: `https://api.paystack.co/transaction/verify/${reference}`,
                headers: {
                    "Content-Type": "application/json",
                    Authorization: `Bearer ${process.env.PAYSTACK_SECRET_KEY}`
                },
            });
            const { status, amount, gateway_response, requested_amount, channel, metadata, message } = verify.data.data
            console.log(metadata, 'metadata');
            const { email, first_name, last_name, DOB } = metadata.customer
            console.log(status, 'status', 'acceptance_fee');
            if (status !== 'success') return badRequest(res, gateway_response)

            if (amount !== requested_amount) throw new Error('Invalid payment');

            const ref = reference as string
            await this.pay_refAdapter.DBSavePaymentRef(email, ref, metadata.fee);

            const replacements = {
                userName: `${first_name} ${last_name}`,
                companyName: process.env.COMPANY_NAME || 'TNM Media Academy',
                orderID: ref,
                orderDate: new Date(),
                paymentMethod: channel,
                shippingAddress: "",
                deliveryDate: "3 days",
                contactInfo: 'tnm_contact',
                supportLink: process.env.SUPPORT_LINK!
            };

            // Get email content with dynamic data
            const emailContent = generateContent('/mails/receipt.html', replacements);
            sendMail({ to: email, type: 'html', subject: 'TNM Acceptance fee', content: emailContent });

            // crete sudent_fee data and update aspi data
            if (status) {
                await this.feeAdapter.DBCreateAspirantFee(auth_user.id, metadata.fee, amount / 100, stream, ref, session);
                await this.aspirantAdapter.DBUpdateAcceptancePaymentStatus(auth_user.id, auth_user.jamb_reg);
            }

            return success(res, status);
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    public getSchoolFee = async (req: Request, res: Response) => {
        try {
            const { auth_user, session } = req.body;
            const { split } = req.query;
            // const stream = auth_user.stream || Stream.STANDARD
            const fee = await this.feeAdapter.DBGetSchoolFee(auth_user.stream, session.id);

            const spt = split === 'true'
            if (spt && fee.upfront) {
                console.log('should')
                fee.amount = 60 / 100 * fee.amount;
            }

            return success(res, fee);
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    public paySchoolFeeAndIDCard = async (req: Request, res: Response) => {
        try {
            const { auth_user, session } = req.body;
            const { email, first_name, last_name, DOB } = auth_user;
            const { programme_id, stream } = req.params as any;
            const { split } = req.query as unknown as { split: boolean }

            // const strm = stream || Stream.STANDARD
            // const category = (stream == 'basic' || stream == 'advanced') ? Category.PROFESSIONAL : Category.STANDARD
            const category = Object.values(NID_Stream).includes(auth_user.stream) ? Category.NID : Category.PROFESSIONAL;
            let fee = await this.feeAdapter.DBGetSchoolFee(auth_user.stream ?? stream, session.id);

            const card_fee = await this.feeAdapter.DBGetIDCardFee(session.id, category)

            // let receipt: { [key: string]: object } = {};
            let amount;

            const transactionID = generateTransactionID();

            fee[transactionID] = transactionID;

            if (split && fee.upfront) {
                // calculate 60% of it;
                amount = 0.6 * fee.amount;
                fee = {
                    ...fee,
                    amount_paid: amount,
                    amount_due: fee.amount,
                    balance: fee.amount - amount
                }
            } else {
                amount = fee.amount;
            }

            amount += card_fee.amount;
            amount = (amount + calculateCharges(amount)) * 100

            delete auth_user.o_level
            delete auth_user.jamb_result
            delete auth_user.adm_letter;

            const metadata = {
                fee,
                card_fee,
                customer: { first_name, last_name, email, DOB },
                stream: auth_user.stream ?? stream,
                invoice_type: Invoice_Type.TUITION,
                auth_user
            };
            const fee_ids = [fee.id, card_fee.id];
            // const callback_url = stream ? `${process.env.TNM_CLIENT_URL}/aspirant/professional/fee/school_fee/verify?fee_ids=${fee_ids}&stream=${stream}` :
            //     `${process.env.TNM_CLIENT_URL}/aspirant/nid/fee/school_fee/verify?fee_ids=${fee_ids}`
            const callback_url = stream ? `${process.env.TNM_CLIENT_URL}/` :
                `${process.env.TNM_CLIENT_URL}/aspirant/nid`

            const payment = await axios({
                method: 'POST',
                url: `https://api.paystack.co/transaction/initialize`,
                headers: {
                    "Content-Type": "application/json",
                    Authorization: `Bearer ${process.env.PAYSTACK_SECRET_KEY}`
                },
                data: { email, amount: Math.ceil(amount), reference: transactionID, metadata, callback_url },
            })

            console.log(transactionID)

            // return success(res, { url: payment.data.data.authorization_url, ref: transactionID, fee_id: fee.id });
            return success(res, { url: '#', ref: transactionID, fee_id: fee.id });


            // return success(res, { url: payment.data.data.authorization_url, ref: transactionID, fee_ids });
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    public processTution = async (session: any, amount: number, metadata: any, reference: string) => {
        try {
            const { auth_user, stream, customer } = metadata;
            const { email, first_name, last_name, DOB } = customer;

            // const strm = metadata.stream || Stream.STANDARD
            const programme = await this.studentAdapter.getProgrammeByStream(stream);
            const paid = metadata.fee.amount_due === metadata.fee.amount_paid;
            const diet = await this.studentAdapter.getDietById(session.current_diet.id)
            const adm_no = await generateRegNo(programme, diet);
            const password = await encodeLastName(last_name)
            // const level = stream != Stream.STANDARD ? Level._13 : Level._100
            const level =  Object.values(NID_Stream).includes(auth_user.stream) ? Level._100 : Level._13
            const student = await this.studentAdapter.DBCreateStudent(auth_user, password, adm_no, paid, level, diet.id)

            // await this.feeAdapter.DBCreateStudentFee(student, metadata.fee, (amount / 100) - metadata.card_fee.amount, strm, reference, programme.id, session, false);
            await this.feeAdapter.DBCreateStudentFee(student, metadata.fee, metadata.fee.amount - metadata.card_fee.amount, stream, reference, programme.id, session, false);
            await this.feeAdapter.DBCreateStudentFee(student, metadata.card_fee, metadata.card_fee.amount, stream, reference, programme.id, session, false);

            // transfer all fee table to student_fee table
            // delete user from aspirant table
            await this.aspirantAdapter.DBTransferReceipt(auth_user.id, student, stream)
            // await this.aspirantAdapter.DBRemoveAspirant(auth_user.id, auth_user.email, stream, student);
            await this.aspirantAdapter.DBCreateEnrollmentV2(auth_user.id, 
                { 
                    stream: auth_user.stream,
                    entry_session: session.id,
                    diet_id: diet.id,
                    programme_id: programme.id,
                    student_id: student,
                    session_id: session.id
                },
                programme.duration
            );

            console.log({ to: email, type: 'html', subject: 'Congratulations!!', content: `Your admission number is ${adm_no}. Kindly login with the last part of the admission number as the matric number and your surname as the passwoord` })
            sendMail({ to: email, type: 'html', subject: 'Congratulations!!', content: `Your admission number is ${adm_no}. Kindly login with the last part of the admission number as the matric number and your surname as the passwoord` });
        } catch (error) {
            throw error;
        }
    }

    public verifySchoolFeeAndIDCard = async (req: Request, res: Response) => {
        try {
            const { session, auth_user } = req.body;
            const { stream } = req.params as any;
            const { reference } = req.query;

            const verify = await axios({
                method: 'GET',
                url: `https://api.paystack.co/transaction/verify/${reference}`,
                headers: {
                    "Content-Type": "application/json",
                    Authorization: `Bearer ${process.env.PAYSTACK_SECRET_KEY}`
                },
            });
            const { status, amount, gateway_response, requested_amount, channel, metadata, message } = verify.data.data
            console.log(metadata, 'metadata');
            const { email, first_name, last_name, DOB } = metadata.customer
            console.log(status, 'status', 'school_fee');
            if (status !== 'success') return badRequest(res, gateway_response)

            if (amount !== requested_amount) throw new Error('Invalid payment');

            const ref = reference as string
            await this.pay_refAdapter.DBSavePaymentRef(email, ref, { fee: metadata.fee, card_fee: metadata.card_fee });

            const replacements = {
                userName: `${first_name} ${last_name}`,
                companyName: process.env.COMPANY_NAME || 'TNM Media Academy',
                orderID: ref,
                orderDate: new Date(),
                paymentMethod: channel,
                shippingAddress: "",
                deliveryDate: "3 days",
                contactInfo: 'tnm_contact',
                supportLink: process.env.SUPPORT_LINK!
            };

            // Get email content with dynamic data
            const emailContent = generateContent('/mails/receipt.html', replacements);
            sendMail({ to: email, type: 'html', subject: 'TNM Acceptance fee', content: emailContent });

            // crete sudent_fee data
            if (status) {
                // const strm = stream || Stream.STANDARD
                const programme = await this.studentAdapter.getProgrammeByStream(stream);
                const paid = metadata.fee.amount_due === metadata.fee.amount_paid;
                const diet = await this.studentAdapter.getDietById(session.current_diet.id)
                const adm_no = await generateRegNo(programme, diet);
                const password = await encodeLastName(last_name);
                const level =  Object.values(NID_Stream).includes(auth_user.stream) ? Level._100 : Level._13
                const student_id = await this.studentAdapter.DBCreateStudent(auth_user, password, adm_no, paid, diet.id)

                // await this.feeAdapter.DBCreateAspirantFee(auth_user.id, metadata.fee, metadata.fee.amount, strm, ref, session);
                // await this.feeAdapter.DBCreateAspirantFee(auth_user.id, metadata.card_fee, metadata.card_fee.amount, strm, ref, session);
                // await this.feeAdapter.DBCreateStudentFee(student_id, metadata.fee, (amount / 100) - metadata.card_fee.amount, strm, ref, programme.id, session, false);
                await this.feeAdapter.DBCreateStudentFee(student_id, metadata.fee, metadata.fee.amount - metadata.card_fee.amount, stream, ref, programme.id, session, false);
                await this.feeAdapter.DBCreateStudentFee(student_id, metadata.card_fee, metadata.card_fee.amount, stream, ref, programme.id, session, false);

                // transfer all fee table to student_fee table
                // delete user from aspirant table

                console.log({ to: email, type: 'html', subject: 'Congratulations!!', content: `Your admission number is ${adm_no}. Kindly login with the last part of the admission number as the matric number and your surname as the passwoord` })
                sendMail({ to: email, type: 'html', subject: 'Congratulations!!', content: `Your admission number is ${adm_no}. Kindly login with the last part of the admission number as the matric number and your surname as the passwoord` });
            }

            return success(res, status);
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    public payAcceptanceSchoolFeeAndIDCard = async (req: Request, res: Response) => {
        try {
            const { auth_user, session } = req.body;
            const { email, first_name, last_name, DOB } = auth_user;
            const { programme_id, stream } = req.params as any;
            const { split } = req.query as unknown as { split: boolean }

            // const strm = stream || Stream.STANDARD
            let fee = await this.feeAdapter.DBGetSchoolFee(auth_user.stream ?? stream, session.id);

            // const category = (stream == 'basic' || stream == 'advanced') ? Category.PROFESSIONAL : Category.STANDARD
            // const category = Object.values(NID_Stream).includes(auth_user.stream) ? Category.NID : Category.PROFESSIONAL;
            // const category = (auth_user.stream === PROF_Stream.BASIC || auth_user.stream === PROF_Stream.ADVANCED) ? Category.PROFESSIONAL : auth_user.stream === NID_Stream.FULL ?  Category.FULL : Category.PART


            const accp_cat = (auth_user.stream === PROF_Stream.BASIC || auth_user.stream === PROF_Stream.ADVANCED) ? Category.PROFESSIONAL : auth_user.stream === NID_Stream.FULL ?  Category.FULL : Category.PART

            const card_fee = await this.feeAdapter.DBGetIDCardFee(session.id, Object.values(NID_Stream).includes(auth_user.stream) ? Category.NID : Category.PROFESSIONAL)
            const acceptance_fee = await this.feeAdapter.DBGetAcceptanceFee(session.id, accp_cat);

            // let receipt: { [key: string]: object } = {};
            let amount;

            const transactionID = generateTransactionID();

            fee[transactionID] = transactionID;

            if (split && fee.upfront) {
                // calculate 60% of it;
                // amount = 60 / 100 * (fee.amount *  100);
                amount = 0.6 * fee.amount;
                fee = {
                    ...fee,
                    amount_paid: amount,
                    amount_due: fee.amount,
                    balance: fee.amount - amount
                }
            } else {
                amount = fee.amount;
            }

            // amount += (card_fee.amount * 100)
            amount = amount + card_fee.amount + acceptance_fee.amount
            amount = (amount + calculateCharges(amount)) * 100

            delete auth_user.o_level
            delete auth_user.jamb_result
            delete auth_user.adm_letter;

            const metadata = {
                fee,
                card_fee,
                acceptance_fee,
                customer: { first_name, last_name, email, DOB },
                stream: auth_user.stream ?? stream,
                invoice_type: Invoice_Type.ACCEPTANCE_TUTION,
                auth_user
            };
            const fee_ids = [fee.id, card_fee.id, acceptance_fee.id];

            // const callback_url = stream ? `${process.env.TNM_CLIENT_URL}/aspirant/professional/fee/all_fee/verify?fee_ids=${fee_ids}&stream=${stream}` :
            //     `${process.env.TNM_CLIENT_URL}/aspirant/nid/fee/all_fee/verify?fee_ids=${fee_ids}`
            const callback_url = stream ? `${process.env.TNM_CLIENT_URL}/` :
                `${process.env.TNM_CLIENT_URL}/`

            const payment = await axios({
                method: 'POST',
                url: `https://api.paystack.co/transaction/initialize`,
                headers: {
                    "Content-Type": "application/json",
                    Authorization: `Bearer ${process.env.PAYSTACK_SECRET_KEY}`
                },
                data: { email, amount: Math.ceil(amount), reference: transactionID, metadata, callback_url },
            })

            console.log(transactionID)

            // return success(res, { url: payment.data.data.authorization_url, ref: transactionID, fee_id: fee.id });

            // return success(res, { url: payment.data.data.authorization_url, ref: transactionID, fee_ids });
            return success(res, { url: '#', ref: transactionID, fee_ids });
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    public processAcceptanceAndTutionFee = async (session: any, amount: number, metadata: any, reference: string) => {
        try {
            const { auth_user, stream, customer } = metadata;
            const { email, first_name, last_name, DOB } = customer;

            // const strm = stream || Stream.STANDARD;
            const paid = metadata.fee.amount_due === metadata.fee.amount_paid;
            const programme = await this.studentAdapter.getProgrammeByStream(stream);
            const diet = await this.studentAdapter.getDietById(session.current_diet.id)
            const adm_no = await generateRegNo(programme, diet);
            const password = await encodeLastName(last_name)
            const level = stream == Stream.BASIC || stream == Stream.ADVANCED ? Level._13 : Level._100
            const student = await this.studentAdapter.DBCreateStudent(auth_user, password, adm_no, paid, level)

            // await this.feeAdapter.DBCreateAspirantFee(auth_user.id, metadata.fee, metadata.fee.amount, strm, ref, session);
            // await this.feeAdapter.DBCreateAspirantFee(auth_user.id, metadata.card_fee, metadata.card_fee.amount, strm, ref, session);
            // await this.feeAdapter.DBCreateAspirantFee(auth_user.id, metadata.acceptance_fee, metadata.acceptance_fee.amount, strm, ref, session);
            const amount_paid = metadata.fee.amount - (Number(metadata.card_fee.amount) + Number(metadata.acceptance_fee.amount))
            // const amount_paid = (amount / 100) - (Number(metadata.card_fee.amount) + Number(metadata.acceptance_fee.amount))
            await this.feeAdapter.DBCreateStudentFee(student, metadata.fee, amount_paid, stream, reference, programme.id, session, false);
            await this.feeAdapter.DBCreateStudentFee(student, metadata.card_fee, metadata.card_fee.amount, stream, reference, programme.id, session, false);
            await this.feeAdapter.DBCreateStudentFee(student, metadata.acceptance_fee, metadata.acceptance_fee.amount, stream, reference, programme.id, session, false);

            // transfer all fee table to student_fee table;
            await this.aspirantAdapter.DBTransferReceipt(auth_user.id, student, stream)
            // await this.aspirantAdapter.DBRemoveAspirant(auth_user.id, auth_user.email, stream, student);
            
            await this.aspirantAdapter.DBCreateEnrollmentV2(auth_user.id,
                { 
                    stream: auth_user.stream,
                    entry_session: session.id,
                    diet_id: diet.id,
                    programme_id: programme.id,
                    student_id: student,
                    session_id: session.id
                },
                programme.duration,
            );

            sendMail({ to: email, type: 'html', subject: 'Congratulations!!', content: `Your admission number is ${adm_no}. Kindly login with the last part of the admission number as the matric number and your surname as the passwoord` });
        } catch (error) {
            throw error;
        }
    }

    public verifyAcceptanceSchoolFeeAndIDCard = async (req: Request, res: Response) => {
        try {
            const { session, auth_user } = req.body;
            const { stream } = req.params as any;
            const { reference } = req.query;

            const verify = await axios({
                method: 'GET',
                url: `https://api.paystack.co/transaction/verify/${reference}`,
                headers: {
                    "Content-Type": "application/json",
                    Authorization: `Bearer ${process.env.PAYSTACK_SECRET_KEY}`
                },
            });
            const { status, amount, gateway_response, requested_amount, channel, metadata, message } = verify.data.data
            console.log(metadata, 'metadata');
            const { email, first_name, last_name, DOB } = metadata.customer
            console.log(status, 'status', 'all_fee');
            if (status !== 'success') return badRequest(res, gateway_response)

            if (amount !== requested_amount) throw new Error('Invalid payment');

            const ref = reference as string
            await this.pay_refAdapter.DBSavePaymentRef(email, ref, { fee: metadata.fee, card_fee: metadata.card_fee, acceptance_fee: metadata.acceptance_fee });

            const replacements = {
                userName: `${first_name} ${last_name}`,
                companyName: process.env.COMPANY_NAME || 'TNM Media Academy',
                orderID: ref,
                orderDate: new Date(),
                paymentMethod: channel,
                shippingAddress: "",
                deliveryDate: "3 days",
                contactInfo: 'tnm_contact',
                supportLink: process.env.SUPPORT_LINK!
            };

            // Get email content with dynamic data
            const emailContent = generateContent('/mails/receipt.html', replacements);
            sendMail({ to: email, type: 'html', subject: 'TNM Acceptance fee', content: emailContent });

            // crete sudent_fee data
            if (status) {
                // const strm = stream || Stream.STANDARD;

                const paid = metadata.fee.amount_due === metadata.fee.amount_paid;
                const programme = await this.studentAdapter.getProgrammeByStream(stream);
                const diet = await this.studentAdapter.getDietById(session.current_diet.id)
                const adm_no = await generateRegNo(programme, diet);
                const password = await encodeLastName(last_name);
                const level = stream == Stream.BASIC || stream == Stream.ADVANCED ? Level._13 : Level._100
                const student_id = await this.studentAdapter.DBCreateStudent(auth_user, password, adm_no, paid, level)

                // await this.feeAdapter.DBCreateAspirantFee(auth_user.id, metadata.fee, metadata.fee.amount, strm, ref, session);
                // await this.feeAdapter.DBCreateAspirantFee(auth_user.id, metadata.card_fee, metadata.card_fee.amount, strm, ref, session);
                // await this.feeAdapter.DBCreateAspirantFee(auth_user.id, metadata.acceptance_fee, metadata.acceptance_fee.amount, strm, ref, session);
                const amount_paid = metadata.fee.amount - (Number(metadata.card_fee.amount) + Number(metadata.acceptance_fee.amount))
                await this.feeAdapter.DBCreateStudentFee(student_id, metadata.fee, amount_paid, stream, ref, programme.id, session, false);
                await this.feeAdapter.DBCreateStudentFee(student_id, metadata.card_fee, metadata.card_fee.amount, stream, ref, programme.id, session, false);
                await this.feeAdapter.DBCreateStudentFee(student_id, metadata.acceptance_fee, metadata.acceptance_fee.amount, stream, ref, programme.id, session, false);

                // transfer all fee table to student_fee table;
                sendMail({ to: email, type: 'html', subject: 'Congratulations!!', content: `Your admission number is ${adm_no}. Kindly login with the last part of the admission number as the matric number and your surname as the passwoord` });
            }

            return success(res, status);
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    // STUDENTTTTTTTT

    public getOutstandingFee = async (req: Request, res: Response) => {
        try {
            const { session, auth_user } = req.body;
            const { returning, split } = req.params as unknown as { returning: boolean, split: boolean };

            const data = await this.studentAdapter.DBGetEnrollmentData(auth_user.id);

            // if(data.session_id === session.id), that means it's not a returning but wants to pay balance for the current session
            const residual = data.session_id === session.id
            const fees = await this.feeAdapter.DBGetOutstandingFee(auth_user.id, session.id, auth_user.stream, residual);
            console.log(fees)

            // const spt = split === true
            // if (spt && fee.upfront) {
            //     console.log('should')
            //     fee.amount = 60 / 100 * fee.amount;
            // }
            return success(res, fees)
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    public payOutstandingFee = async (req: Request, res: Response) => {
        try {
            const { session, auth_user } = req.body;
            const { first_name, last_name, email, stream } = auth_user;

            const data = await this.studentAdapter.DBGetEnrollmentData(auth_user.id);

            const residual = data.session_id === session.id
            const fees = await this.feeAdapter.DBGetOutstandingFee(auth_user.id, session.id, auth_user.stream, residual);

            let amount = 0;
            // let receipt: { [key: string]: object } = {};
            for (let fee of fees) {
                amount += fee.amount;

                // receipt[fee.description] = fee;
            }
            amount = (amount + calculateCharges(amount)) * 100;

            const transactionID = generateTransactionID();

            fees[transactionID] = transactionID;

            const metadata = {
                fees,
                customer: { first_name, last_name, email },
                stream
            };

            const payment = await axios({
                method: 'POST',
                url: `https://api.paystack.co/transaction/initialize`,
                headers: {
                    "Content-Type": "application/json",
                    Authorization: `Bearer ${process.env.PAYSTACK_SECRET_KEY}`
                },
                data: { email, amount: Math.ceil(amount), reference: transactionID, metadata, callback_url: `${process.env.TNM_CLIENT_URL}/student/fees/verify?fee_ids=${fees.map(fee => fee.id)}&stream=${stream}&residual=${residual}` },
            })

            return success(res, payment.data.data.authorization_url);
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    public verifyOutstandingFeePayment = async (req: Request, res: Response) => {
        try {
            const { auth_user, session } = req.body;
            const { reference, stream, residual } = req.query as unknown as { reference: string, stream: Stream, residual: boolean }

            const { id, email } = auth_user;

            const verify = await axios({
                method: 'GET',
                url: `https://api.paystack.co/transaction/verify/${reference}`,
                headers: {
                    "Content-Type": "application/json",
                    Authorization: `Bearer ${process.env.PAYSTACK_SECRET_KEY}`
                },
            });

            const { status, amount, gateway_response, requested_amount, channel, metadata, message } = verify.data.data

            console.log(status, 'status')
            if (status !== 'success') return badRequest(res, gateway_response)

            if (amount !== requested_amount) throw new Error('Invalid fund');

            const ref = reference as string
            // await this.pay_refAdapter.DBSavePaymentRef(email, ref, { fees: metadata.fees });

            const replacements = {
                userName: `${auth_user.first_name} ${auth_user.last_name}`,
                companyName: process.env.COMPANY_NAME!,
                orderID: reference,
                orderDate: new Date(),
                paymentMethod: channel,
                shippingAddress: "",
                deliveryDate: "3 days",
                contactInfo: 'cowry_contact',
                supportLink: process.env.SUPPORT_LINK!
            };

            // Get email content with dynamic data
            const emailContent = generateContent('/mails/receipt.html', replacements);
            sendMail({ to: email, type: 'html', subject: 'Please verify your email', content: emailContent });

            if (status) {
                const programme = await this.studentAdapter.getProgrammeByStream(auth_user.stream);
                const resid = residual === true
                for (let fee of metadata.fees) {
                    await this.feeAdapter.DBCreateStudentFee(auth_user.id, fee, fee.amount, auth_user.stream, ref, programme.id, session, resid);
                }
            }

            return successAction(res, status);
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    // W E B H O O K
    public paymentWebhook = async (req: Request, res: Response) => {
        try {
            const hmac = forge.hmac.create();
            hmac.start('sha512', process.env.PAYSTACK_SECRET_KEY!);
            hmac.update(JSON.stringify(req.body));
            const hash = hmac.digest().toHex();

            console.log(hash, 'hash');
            console.log(req.headers['x-paystack-signature'], 'req.headers');

            successAction(res);

            // Retrieve the request's body
            const event = req.body;
            // if(event.event !== 'charge.success') return successAction(res);
            if (hash !== req.headers['x-paystack-signature'] && event.event !== 'charge.success') {
                console.log('Not from paystack');

                // return badRequest(res, null,)
                return
            }


            const { status, amount, gateway_response, requested_amount, channel, metadata, message } = event.data
            if (status !== 'success') return badRequest(res, gateway_response);
            if (amount !== requested_amount) throw new Error('Invalid fund')

            console.log(req.body);

            const reference = event.data.reference;

            // const verify = await axios({
            //     method: 'GET',
            //     url: `https://api.paystack.co/transaction/verify/${reference}`,
            //     headers: {
            //         "Content-Type": "application/json",
            //         Authorization: `Bearer ${process.env.PAYSTACK_SECRET_KEY}`
            //     },
            // });
            console.log(metadata, 'metadata');
            console.log(metadata.invoice_type, 'Webhook fee type to be firedddddddd');

            switch (metadata.invoice_type) {
                case Invoice_Type.FORM:
                    this.processFormPayment(event.session, amount, metadata, reference);
                    break;
                case Invoice_Type.PUTME:
                    this.processPUTMEPayment(event.session, amount, metadata, reference);
                    break;
                case Invoice_Type.PART_FORM:
                    this.processPartApplicationPayment(event.session, amount, metadata, reference);
                    break;
                case Invoice_Type.ACCEPTANCE:
                    this.processAcceptanceFee(event.session, amount, metadata, reference);
                    break;
                case Invoice_Type.TUITION:
                    this.processTution(event.session, amount, metadata, reference);
                    break;
                case Invoice_Type.ACCEPTANCE_TUTION:
                    this.processAcceptanceAndTutionFee(event.session, amount, metadata, reference);
                    break;
                default:
                    console.log('COULD NOT FOUND A MATCHING FEE TYPE');

                    break;
            }


            const { email, first_name, last_name, middle_name, DOB, phone } = metadata.customer

            const ref = reference as string
            // await this.pay_refAdapter.DBSavePaymentRef(email, ref, metadata.fee);

            const replacements = {
                userName: `${first_name} ${last_name}`,
                companyName: process.env.COMPANY_NAME || 'TNM Media Academy',
                orderID: ref,
                orderDate: new Date(),
                paymentMethod: channel,
                shippingAddress: "",
                deliveryDate: "3 days",
                contactInfo: 'tnm_contact',
                supportLink: process.env.SUPPORT_LINK!
            };

            // Get email content with dynamic data
            const emailContent = generateContent('/mails/receipt.html', replacements);
            sendMail({ to: email, type: 'html', subject: 'Payment Receipt', content: emailContent });

        } catch (error) {
            return this.catchError(error, res);
        }
    }
};

export default FeeController;