import { Service } from "typedi";
import knex, { BaseAdapter } from ".";
import { ETables } from "../../types/db.types";
import { getWeek } from "date-fns";
import * as bcrypt from "bcryptjs";
import { Category, NID_Stream, PROGRAMMES, Permisions, Review, Status, Stream } from "../../types/enum";
import { excelSerialDateToJSDate } from "../../Utils/factory";

@Service()
class AdminAdapter extends BaseAdapter {

  public DBGetDashboardData = async () => {
    try {

      //       const students = await knex(ETables.STUDENT).select('stream').groupBy("stream").count('* as student_count')
      //       const total_students = students.map((item: any) => {
      //         return { [item.stream]: parseInt(item.student_count) };
      //     });
      // console.log(total_students)


      // const total_students = await knex(ETables.STUDENT).count('* as student_count').groupBy("stream");
      const total_students = knex(ETables.STUDENT).count('* as student_count');
      // console.log(total_students)
      const prof_aspirants = knex(ETables.PROF_ASPIRANT).where('status', 'pending').count('* as professional_aspirants')

      const aspirants = knex(ETables.ASPIRANT).where('status', 'pending').count('* as aspirants_count');

      const courses = knex(ETables.COURSES).count('* as courses_count');

      const diets = knex(ETables.DIET).count('* as total_diet');

      const sessions = knex(ETables.DIET).count('* as total_sessions');

      const [ [{ student_count }], [{ professional_aspirants }], [{ aspirants_count }], [{ courses_count }], [{ total_diet }], [{ total_sessions }] ] = await Promise.all([total_students, prof_aspirants, aspirants, courses, diets, sessions]);
      // console.log({ student_count, professional_aspirants, aspirants_count, courses_count })
      return { student_count, professional_aspirants, aspirants_count, courses_count, total_diet, total_sessions };
    } catch (error) {
      return this.catchError(error);
    }
  }

  public DBCreateAdmin = async (data: {
    email: string,
    password: string,
    role_id: number,
    name: string,
    // username: string,
  }) => {
    try {
      let [user] = await knex.table(ETables.ADMIN).insert(data);

      return user;
    } catch (error) {
      return this.catchError(error);
    }
  }

  public DBGetAdmins = async () => {
    try {
      const admins = await knex.select([
        "admin.id as id",
        "admin.email as email",
        "admin.role_id as role_id",
        "admin.name as name",
        "role.title as role",
        "role.permissions as permissions"
      ])
        .from({ admin: ETables.ADMIN })
        .join({ role: ETables.ADMIN_ROLE }, 'role.id', 'admin.role_id')

      return admins;
    } catch (error) {
      return this.catchError(error);
    }
  }

  public DBEditAdmin = async (id: number, data: { name?: string, role_id?: number, email?: string }) => {
    try {
      await knex(ETables.ADMIN).update(data).where('id', id);
    } catch (error) {
      return this.catchError(error);
    }
  }

  public DBCreateSuperAdmin = async () => {
    try {
      const password = bcrypt.hashSync(process.env.ADMIN_PASS!, 10);
      await knex.transaction(async trx => {
        let role = await trx.select("*").from(ETables.ADMIN_ROLE).where("title", "Super Admin").first();
        if (!role) {
          [role] = await trx(ETables.ADMIN_ROLE).insert({
            title: "Super Admin",
            description: "All priviledged admin",
            permissions: Permisions.SUDO
          })

          await trx(ETables.ADMIN).insert({
            password, email: "admin@tnmmediaacademy.com", role_id: role, name: "Super Admin",
          })
        }
      })
    } catch (error) {
      return this.catchError(error);
    }
  }

  public DBCreateAdminRole = async (data: { title: string, description: string, permissions: string }) => {
    try {
      const [role] = await knex.table(ETables.ADMIN_ROLE).insert(data)

      return role;
    } catch (error) {
      return this.catchError(error);
    }
  }

  public DBGetRoles = async () => {
    try {
      const roles = await knex.select("*").from(ETables.ADMIN_ROLE).whereNot('permissions', '*');

      return roles;
    } catch (error) {
      return this.catchError(error);
    }
  }

  public DBGetMe = async (id: number) => {
    try {
      const me = await knex.select([
        'adm.*',
        'role.title as role',
        'role.permissions as permissions'
      ])
        .from({ adm: ETables.ADMIN })
        .join({ role: ETables.ADMIN_ROLE }, 'role.id', 'adm.role_id')
        .where('adm.id', id).first();

      return me;
    } catch (error) {
      return this.catchError(error);
    }
  }

  public DBGetSessions = async () => {
    try {
      const sessions = await knex.select('*').from(ETables.ACADEMIC_SESSION);

      return sessions;
    } catch (error) {
      return this.catchError(error);
    }
  }

  public DBGetAllProgrammes = async () => {
    try {
      const programmes = await knex.select('*').from(ETables.PROGRAMME);

      return programmes;
    } catch (error) {
      return this.catchError(error);
    }
  }

  public DBGetProgramme = async (prog_level: Stream) => {
    try {
      const programme = await knex.select('*').from(ETables.PROGRAMME).where("prog_level", prog_level).first();

      return programme;
    } catch (error) {
      return this.catchError(error);
    }
  }

  public DBGetCourses = async (stream: Stream) => {
    try {
      const courses = await knex.select('*').from(ETables.COURSES).where("stream", stream)

      return courses;
    } catch (error) {
      return this.catchError(error);
    }
  }

  public DBGetACourse = async (stream: Stream, id: number) => {
    try {
      const course = await knex.select('*').from(ETables.COURSES).where("stream", stream).andWhere('id', id).first();

      return course;
    } catch (error) {
      return this.catchError(error);
    }
  }

  public DBGetAssignedCourses = async (admin_id: number, stream: Stream) => {
    try {
      const assigned_courses = await knex.select([
        "ass.*",
        "course.code as code",
        "course.title as title",
        "course.description as description",
        "course.unit as unit",
        "course.type as type",
        "course.level as level",
        "course.semester as semester",
      ])
        .from({ ass: ETables.ASSIGNED_COURSES })
        .join({ course: ETables.COURSES }, 'course.id', 'ass.course_id')
        .where("ass.stream", stream).andWhere('admin_id', admin_id)

      return assigned_courses;
    } catch (error) {
      return this.catchError(error);
    }
  }

  public DBAssignStaff = async (stream: Stream, id: number, admin_ids: number[], session_id: number) => {
    try {
      const obj = [];

      for (let admin_id of admin_ids) {
        obj.push({
          admin_id,
          course_id: id,
          stream,
          session_id
        });
      }

      await knex(ETables.ASSIGNED_COURSES).insert(obj);

    } catch (error) {
      return this.catchError(error);
    }
  }

  public DBGetAssignedStaff = async (stream: Stream, id: number, session_id: number) => {
    try {
      const staff = await knex.select([
        "ass.*",
        'admin.name as name',
        'role.title as role'
      ])
        .from({ ass: ETables.ASSIGNED_COURSES })
        .join({ admin: ETables.ADMIN }, 'admin.id', 'ass.admin_id')
        .join({ role: ETables.ADMIN_ROLE }, 'role.id', 'admin.role_id')
        .where('stream', stream).andWhere('course_id', id).andWhere('session_id', session_id).first();

      return staff;
    } catch (error) {
      return this.catchError(error);
    }
  }

  public DBGetStudents = async () => {
    try {
      const students = await knex.select([
        "enrl.*",
        "std.first_name as first_name",
        "std.last_name as last_name",
        "std.email as email",
        "std.adm_no as adm_no",
        "std.level as level",
        // "std.phone as phone",
        "std.DOB as DOB",
        "prg.name as programme",
        "prg.duration as duration",
        "prg.stream as stream",
        "diet.number as diet"
      ])
        .from({ enrl: ETables.ENROLLMENT })
        .join({ std: ETables.STUDENT }, "std.id", "enrl.student_id")
        .leftJoin({ prg: ETables.PROGRAMME }, "prg.id", "enrl.programme_id")
        .leftJoin({ diet: ETables.DIET }, "diet.id", "enrl.diet_id")
      console.log(students[0])
      return students;
    } catch (error) {
      return this.catchError(error);
    }
  }

  public DBGetPendingApplications = async (programme: Stream) => {
    try {
      let applicatiions;

      // if (programme === Stream.STANDARD) {
      if (Object.values(NID_Stream as any).includes(programme)) {
        applicatiions = await knex.select(['aspi.*'])
        .from({aspi: ETables.ASPIRANT})
        .join({ prog: ETables.PROGRAMME }, 'prog.id', 'aspi.programme_id')
        .where("status", Status.PENDING).andWhere('prog.stream', programme);
      } else {
        applicatiions = await knex.select(['aspi.*'])
        .from({aspi: ETables.PROF_ASPIRANT})
        .join({ prog: ETables.PROGRAMME }, 'prog.id', 'aspi.programme_id')
        .where("status", Status.PENDING).andWhere('prog.stream', programme);
      }

      return applicatiions;
    } catch (error) {
      return this.catchError(error);
    }
  }

  public DBGetApplication = async (id: number, programme: Stream) => {
    try {
      let applicatiions;

      const select = [
        "std.*",
        "prog.name as programme",
        "prog.description as description",
        "prog.duration as duration",
        "prog.stream as stream",
        "prog.open_to_application as open_to_application",
      ]

      // if (programme === Stream.STANDARD) {
      if (Object.values(NID_Stream as any).includes(programme)) {
        applicatiions = await knex.select(select)
          .from({ std: ETables.ASPIRANT })
          .leftJoin({ prog: ETables.PROGRAMME }, 'prog.id', 'std.programme_id')
          .where("status", Status.PENDING).andWhere("std.id", id).first();
      } else {
        applicatiions = await knex.select(select)
          .from({ std: ETables.PROF_ASPIRANT })
          .join({ prog: ETables.PROGRAMME }, 'prog.id', 'std.programme_id')
          .where("status", Status.PENDING).andWhere("std.id", id).first();
      }

      return applicatiions;
    } catch (error) {
      return this.catchError(error);
    }
  }

  public DBReviewApplication = async (id: number, programme: Stream, review: { programme_id: number, status: Review, }) => {
    try {
      // if (programme === Stream.STANDARD) 
      if (Object.values(NID_Stream as any).includes(programme)) {
        await knex(ETables.ASPIRANT).update(review).where('id', id).onConflict().merge()
      } else {
        await knex(ETables.PROF_ASPIRANT).update(review).where('id', id).onConflict().merge()
      }
    } catch (error) {
      return this.catchError(error);
    }
  }

  public DBGetCourseResult = async (id: number, session_id: number) => {
    try {
      const results = await knex.select([
        "reg.*",
        "std.first_name as first_name",
        "std.last_name as last_name",
        "std.email as email",
        "std.adm_no as adm_no",
        "std.matric_no as matric_no",
      ])
        .from({ reg: ETables.COURSE_REG })
        .join({ std: ETables.STUDENT }, 'std.id', 'reg.student_id')
        .where("course_id", id).andWhere("session_id", session_id);

      return results;
    } catch (error) {
      return this.catchError(error);
    }
  }


















  public DBUploadResult = async (student_id: number, score: number, course_id: number, session_id: number) => {
    try {
      await knex(ETables.COURSE_REG).update({ grade: score }).where('student_id', student_id).andWhere({ course_id, session_id })
    } catch (error) {
      return this.catchError(error);
    }
  }

  public DBGetSelectedStudents = async (matric_nos: string[]) => {
    try {
      const students = await knex.select('*').from(ETables.STUDENT).whereIn('matric_no', matric_nos);

      return students;
    } catch (error) {
      return this.catchError(error);
    }
  }

  public DBUploadResultV2 = async (updates: { student_id: number, CA: number, exam: number, grade: number }[], course_id: number, session_id: number, level: number) => {
    try {
      // let caseStatement = 'CASE';

      // updates.forEach(update => {
      //   caseStatement += knex.raw(` WHEN student_id = ? AND course_id = ? THEN ?`, [update.student_id, update.course_id, update.grade]).toString();
      // });

      // caseStatement += ' END';

      // let updateQuery = knex(ETables.COURSE_REG)
      //   .update({
      //     grade: knex.raw(caseStatement)
      //   })
      //   .where(knex.raw(`(student_id, course_id) IN (${updates.map(update => `(${update.student_id}, ${update.course_id})`).join(', ')})`))
      //   .toString();

      // knex.raw(updateQuery)
      //   .then(() => console.log("Batch update completed"))
      //   .catch(err => console.error(err));


      console.log(updates, 'updates')

      // Preparing the CASE statements for both ca and exam fields
      let caCaseStatement = 'CASE student_id ';
      let examCaseStatement = 'CASE student_id ';

      // Loop through each update to append conditions for both fields
      updates.forEach(update => {
        caCaseStatement += knex.raw(`WHEN ? THEN ? `, [update.student_id, update.CA]);
        examCaseStatement += knex.raw(`WHEN ? THEN ? `, [update.student_id, update.exam]);
      });

      // Closing each CASE statement
      caCaseStatement += 'ELSE grade END';
      examCaseStatement += 'ELSE exam END';

      // Executing the batch update with both CASE statements
      knex(ETables.COURSE_REG)
        .update({
          CA: knex.raw(caCaseStatement),
          exam: knex.raw(examCaseStatement),
          grade: knex.raw('ca + exam')
        })
        .where({
          session_id,
          course_id,
          current_level: level,
        })
        .whereIn('student_id', updates.map(update => update.student_id))
        .then(() => console.log("Batch update completed"))
        .catch(err => console.error(err));


    } catch (error) {
      return this.catchError(error);
    }
  }




























  public DBGetAllFee = async () => {
    try {
      const fees = await knex.select('*').from(ETables.FEE);

      return fees;
    } catch (error) {
      return this.catchError(error);
    }
  }

  public DBGetOtherFee = async () => {
    try {
      const fees = await knex.select('*').from(ETables.OTHER_FEE);

      return fees;
    } catch (error) {
      return this.catchError(error);
    }
  }

  public DBSetFee = async (data: { name: string, amount: number, prog_level: Stream, programme_id: number, category: Category, is_recurrent: boolean, session_id: number }) => {
    try {
      const [fees] = await knex(ETables.FEE).insert({ ...data });

      return fees;
    } catch (error) {
      return this.catchError(error);
    }
  }

  public DBUploadAspirants = async (data: { session_id: number, file: Buffer, mimetype: string, sheetName: string, year: number }, aspirants: any[]) => {
    try {
      // await knex.transaction(async trx => {
      //   const asprnts = [];
      //   for(let aspi of aspirants) {
      //     const obj = {
      //       jamb_reg: aspi.jamb_reg,
      //       email: aspi.email
      //     }

      //     asprnts.push(obj);
      //   };

      //   await trx(ETables.ASPIRANT).insert(aspirants);

      //   await trx(ETables.RESULT).insert({ ...data }); 
      // })
      const asprnts = [];
      for (let aspi of aspirants) {
        const password = bcrypt.hashSync(aspi['Last Name'], 10);

        const obj = {
          jamb_reg: aspi['Jamb Registration'],
          email: aspi['Email'],
          first_name: aspi['First Name'],
          last_name: aspi['Last Name'],
          DOB: excelSerialDateToJSDate(aspi.DOB),
          password
        }

        asprnts.push(obj);
      };

      console.log(aspirants, 'exAspirants');

      await knex(ETables.ASPIRANT).insert(asprnts);

    } catch (error) {
      return this.catchError(error);
    }
  }

  public DBGetAspirants = async () => {
    try {
      const aspirants = await knex.select('*').from(ETables.ASPIRANT);

      return aspirants;
    } catch (error) {
      return this.catchError(error);
    }
  }

  public DBViewAspirant = async (jamb_reg: string) => {
    try {
      const aspirant = await knex.select('*').from(ETables.ASPIRANT).where("jamb_reg", jamb_reg).first();

      return aspirant;
    } catch (error) {
      return this.catchError(error);
    }
  }

  public DBUploadPutmeResult = async (data: { type: string, session_id: number, file: Buffer, mimetype: string, sheetName: string, year: number }, aspirants: any[], results: any[]) => {
    try {
      await knex.transaction(async trx => {
        const reslt = await trx.select('*').from(ETables.RESULT).where({ type: data.type, session_id: data.session_id, year: data.year }).first();

        if (reslt) {
          console.log('result exist ooo')
          await trx(ETables.RESULT).update({ file: data.file }).where({ type: data.type, session_id: data.session_id, year: data.year }).onConflict().merge()
        } else {
          await trx(ETables.RESULT).insert({ ...data });
        }

        for (let aspirant of aspirants) {
          for (let result of results) {
            console.log('updating aspirant')
            if (aspirant.jamb_reg === result.jamb_reg) {
              await trx(ETables.ASPIRANT).update({ grade: result.grade || result.score }).where('jamb_reg', aspirant.jamb_reg).onConflict().merge();
            }
          }
        }

      })

    } catch (error) {
      return this.catchError(error);
    }
  }

  public changeAspirantStatus = async (status: Status, jamb_regs: string[]) => {
    try {

      await knex.table(ETables.ASPIRANT).update({ status }).whereIn("jamb_reg", jamb_regs).onConflict().merge();

    } catch (error) {
      return this.catchError(error);
    }
  }

  public DBGetAcceptedAspirants = async () => {
    try {
      const aspirants = await knex.select('*').from(ETables.ASPIRANT).where('status', 'accepted');

      return aspirants;
    } catch (error) {
      return this.catchError(error);
    }
  }

  public DBGetRejectedAspirants = async () => {
    try {
      const aspirants = await knex.select('*').from(ETables.ASPIRANT).where('status', 'rejected');

      return aspirants;
    } catch (error) {
      return this.catchError(error);
    }
  }

  public DBGetStudentsByProgramme = async (programme_id: number) => {
    try {
      const students = await knex.select([
        "enrl.*",
        "std.first_name as first_name",
        "std.last_name as last_name",
        "std.email as email",
        "std.adm_no as adm_no",
        "std.phone as phone",
        "std.DOB as DOB",
        "std.jamb_reg as jamb_reg",
        "prg.*"
      ])
        .from({ enrl: ETables.ENROLLMENT })
        .join({ std: ETables.STUDENT }, "std.id", "enrl.student_id")
        .leftJoin({ prg: ETables.PROGRAMME }, "prg.id", "enrl.programme_id")
        .where('programme_id', programme_id);

      return students;
    } catch (error) {
      return this.catchError(error);
    }
  }
}

@Service()
export class AdminValidatorAdapter extends BaseAdapter {
  public DBGetUserAndPassword = async (email: string) => {
    try {
      const admin = await knex.select([
        "adm.email as email",
        "adm.id as id",
        "adm.password as password",
        'role.title as role',
        'role.permissions as permissions'
      ])
        .from({ adm: ETables.ADMIN })
        .join({ role: ETables.ADMIN_ROLE }, 'role.id', 'adm.role_id')
        .where("email", email).first();

      return admin;
    } catch (error) {
      return this.catchError(error);
    }
  }

  public IsAValidSession = async (id: number) => {
    try {
      const session = await knex.select('*').from(ETables.ACADEMIC_SESSION).where("id", id).first();

      return session
    } catch (error) {
      return this.catchError(error);
    }
  }

  public DoesProgrammeExist = async (id: number) => {
    try {
      const programme = await knex.select('*').from(ETables.PROGRAMME).where("id", id).first();

      return programme
    } catch (error) {
      return this.catchError(error);
    }
  }

  public CheckProgrammeByStream = async (stream: Stream) => {
    try {
      const programme = await knex.select('*').from(ETables.PROGRAMME).where("stream", stream).first();

      return programme
    } catch (error) {
      return this.catchError(error);
    }
  }

  public DBGetAdmin = async (id: number) => {
    try {
      const admin = await knex.select('*').from(ETables.ADMIN).where("id", id).first();

      return admin
    } catch (error) {
      return this.catchError(error);
    }
  }

  public checkForValidRole = async (role_id: number) => {
    try {
      const role = await knex.select('*').from(ETables.ADMIN_ROLE).where("id", role_id).first();

      return role
    } catch (error) {
      return this.catchError(error);
    }
  }

  public checkProgramByCategory = async (prog_level: Stream) => {
    try {
      const programme = await knex.select('*').from(ETables.PROGRAMME).where("prog_level", prog_level).first();

      return programme
    } catch (error) {
      return this.catchError(error);
    }
  }

  public DBCheckApplication = async (id: number, stream: Stream) => {
    try {
      let application;

      // if (stream === Stream.STANDARD) {
      if (Object.values(NID_Stream as any).includes(stream)) {
        application = await knex.select('*').from(ETables.ASPIRANT).where("id", id).first();
      } else {
        application = await knex.select('*').from(ETables.PROF_ASPIRANT).where("id", id).first();
      }

      return application
    } catch (error) {
      return this.catchError(error);
    }
  }
}

export default AdminAdapter;
