import { Service } from "typedi";
import knex, { BaseAdapter } from ".";
import { ETables } from "../../types/db.types";
import { Gender, NID_Stream, PROF_Stream, Status, Stream } from "../../types/enum";
import { getCompletionDateV2 } from "../../Utils/factory";

@Service()
class AspirantAdapter extends BaseAdapter {
    constructor() {
        super()
    }

    public DBGetProgrammeByStream = async (stream: Stream) => {
        try {
            const programme = await knex.select('*').from(ETables.PROGRAMME).where("stream", stream).first();

            return programme;
        } catch (error) {
            return this.catchError(error);
        }
    }

    public DBGetProfProgrammes = async () => {
        try {
            const programme = await knex.select('*').from(ETables.PROGRAMME).whereNotIn("stream", [Stream.FULL, Stream.PART]);

            return programme;
        } catch (error) {
            return this.catchError(error);
        }
    }

    public DBCreateProfAspirant = async (data: { email: string, first_name: string, last_name: string, password: string, middle_name: string, phone: string }, stream: PROF_Stream, session_id: number, diet_id: number) => {
        try {
            const programme = await knex.select('*').from(ETables.PROGRAMME).where('stream', stream).first();

            // const year = new Date().getUTCFullYear();
            const [aspirant] = await knex(ETables.PROF_ASPIRANT).insert({ ...data, programme_id: programme.id, session_id, diet_id });

            return aspirant;
        } catch (error) {
            return this.catchError(error);
        }
    }

    public DBGetProfAspirant = async (email: string) => {
        try {
            const aspirant = await knex.select([
                "aspi.first_name as first_name",
                "aspi.last_name as last_name",
                "aspi.middle_name as middle_name",
                "aspi.email as email",
                "aspi.qualification as qualification",
                "aspi.workplace as workplace",
                "aspi.years_of_exp as years_of_exp",
                "aspi.certificate as certificate",
                "aspi.paid as paid",
                "aspi.status as status",
                'prog.id as programme_id',
                'prog.name as programme_name',
                'prog.duration as programme_duration',
                'prog.stream as programme_stream',
                'prog.open_to_application as programme_available',
                // 'diet.number as diet',
                // 'diet.year as year',
                // 'diet.session_id as session',
                // 'diet.is_current as is_current_diet',
            ]).from({ aspi: ETables.PROF_ASPIRANT })
                .join({ prog: ETables.PROGRAMME }, 'prog.id', 'aspi.programme_id')
                // .join({ diet: ETables.DIET }, 'diet.id', 'aspi.diet_id')
                .where("email", email).first();

            return aspirant;
        } catch (error) {
            return this.catchError(error);
        }
    }

    public DBCreateApplication = async (data: { first_name: string, last_name: string, middle_name: string, DOB: Date, gender: Gender, qualification: string, years_of_exp: number, workplace: string }, email: string, stream: PROF_Stream) => {
        try {
            const programme = await knex.select(['id']).from(ETables.PROGRAMME).where('stream', stream).first();

            await knex(ETables.PROF_ASPIRANT).update({ ...data, status: Status.PENDING }).where('email', email).andWhere('programme_id', programme.id);

        } catch (error) {
            return this.catchError(error);
        }
    }

    // NID

    public DBPUTMERegistration = async (data: { first_name: string, last_name: string, middle_name: string, email: string, phone: any, password: string, DOB: Date, gender: Gender, jamb_reg: string, o_level_path: Buffer, o_level_mimetype: string, jamb_result_path: string, jamb_result_mimetype: string }, reg_no: string, programme_id: number, session: any) => {
        try {
            const [aspirant] = await knex(ETables.ASPIRANT).insert({ ...data, reg_no, programme_id, session_id: session.id, diet_id: session.current_diet.id });

            return aspirant;
        } catch (error) {
            return this.catchError(error);
        }
    }

    public DBPartApplication = async (data: { first_name: string, last_name: string, middle_name: string, email: string, phone: any, password: string, DOB: Date, gender: Gender, state_of_origin: string, lga: string, address: string, manual_o_level: { subject: string, grade: string }[], o_level_path: string, o_level_mimetype: string, passport_path: string, passport_mimetype: string }, reg_no: string, programme_id: number, session: any) => {
        try {
            const serializedManualOLevel = JSON.stringify(data.manual_o_level);
            const [aspirant] = await knex(ETables.ASPIRANT).insert({ ...data, manual_o_level: serializedManualOLevel, reg_no, programme_id, session_id: session.id, diet_id: session.current_diet.id });

            return aspirant;
        } catch (error) {
            return this.catchError(error);
        }
    }

    public DBGetAspirant = async (jamb_reg: string) => {
        try {
            const aspirant = await knex.select([
                'aspi.*',
                'prog.stream as programme_stream',
            ])
                .from({ aspi: ETables.ASPIRANT }).where('aspi.jamb_reg', jamb_reg)
                .join({ prog: ETables.PROGRAMME }, 'prog.id', 'aspi.programme_id').first();

            return aspirant;
        } catch (error) {
            return this.catchError(error);
        }
    }

    public DBGetAspirantByEmail = async (email: string) => {
        try {
            const aspirant = await knex.select([
                'aspi.*',
                'prog.stream as programme_stream',
            ])
                .from({ aspi: ETables.ASPIRANT }).where('aspi.email', email)
                .join({ prog: ETables.PROGRAMME }, 'prog.id', 'aspi.programme_id').first();

            return aspirant;
        } catch (error) {
            return this.catchError(error);
        }
    }

    // public DBUploadAdmissionLetter = async (jamb_reg: string, data: { adm_letter: Buffer, adm_letter_mimetype: string }) => {
    public DBUploadAdmissionLetter = async (jamb_reg: string, data: { adm_letter_path: string, adm_letter_mimetype: string }) => {
        try {
            const aspirant = await knex.select('*').from(ETables.ASPIRANT).where('jamb_reg', jamb_reg).first();

            await knex(ETables.ASPIRANT).update({ ...data }).where('jamb_reg', jamb_reg).andWhere('id', aspirant.id);
        } catch (error) {
            return this.catchError(error);
        }
    }

    // BOTH

    public DBUpdateAcceptancePaymentStatus = async (id: number, jamb_reg?: string) => {
        console.log(id, 'idDDDDddd')
        try {
            if (!jamb_reg) {
                await knex(ETables.PROF_ASPIRANT).update({ paid: true }).where('id', id);
            } else {
                await knex(ETables.ASPIRANT).update({ paid: true }).where('id', id).andWhere('jamb_reg', jamb_reg);
            }
        } catch (error) {
            return this.catchError(error);
        }
    }

    // public DBCreateStudent = async ({}) => {
    //     try {

    //     } catch (error) {
    //         return this.catchError(error);
    //     }
    // }

    public DBTransferReceipt = async (id: number, student_id: number, stream: Stream) => {
        try {
            let column = '';
            // stream == Stream.STANDARD ? column = 'aspirant_id' : column = 'prof_aspirant_id';
            Object.values(NID_Stream as any).includes(stream) ? column = 'aspirant_id' : column = 'prof_aspirant_id';
            await knex.transaction(async trx => {
                const receipt = await trx.select('*').from(ETables.ASPIRANT_FEE).where(column, id);

                // Check if receipt is empty
                if (receipt.length === 0) {
                    console.error('No records found for the given column and ID:', column, id);
                    throw new Error('No records found to transfer.');
                }

                for (let fee of receipt) {
                    delete fee.id
                    delete fee.prof_aspirant_id;
                    delete fee.aspirant_id;
                    fee.student_id = student_id;
                }

                // Insert into STUDENT_FEE
                await trx(ETables.STUDENT_FEE).insert(receipt);
                console.log(`Successfully inserted ${receipt.length} records into ${ETables.STUDENT_FEE}`);

                // Delete from ASPIRANT_FEE
                const deletedCount = await trx(ETables.ASPIRANT_FEE).where(column, id).delete();
                console.log(`Successfully deleted ${deletedCount} records from ${ETables.ASPIRANT_FEE}`);

            })

        } catch (error) {
            return this.catchError(error);
        }
    }

    // public DBRemoveAspirant = async (id: number, email: string, stream: Stream, student_id: number) => {
    public DBCreateEnrollmentV1 = async (id: number, email: string, stream: Stream, student_id: number) => {
        try {
            // stream == Stream.STANDARD && await knex(ETables.ASPIRANT).where('id', id).andWhere("email", email).delete();
            // stream != Stream.STANDARD && await knex(ETables.PROF_ASPIRANT).where('id', id).andWhere("email", email).delete();

            await knex.transaction(async trx => {
                let data;
                let columns;

                if (stream === Stream.FULL) {
                    columns = [
                        'o_level', 'o_level_mimetype', 'jamb_result', 'jamb_result_mimetype',
                        'adm_letter', 'adm_letter_mimetype'
                    ];
                } else if (stream === Stream.PART) {
                    columns = [
                        'o_level', 'o_level_mimetype',
                        'adm_letter', 'adm_letter_mimetype'
                    ];
                }

                if (columns) {
                    data = await trx.select(columns)
                        .from(ETables.ASPIRANT)
                        .where("id", id)
                        .andWhere("email", email)
                        .first();

                    await trx(ETables.ENROLLMENT)
                        .where("student_id", student_id)
                        .update(data);

                    // Removing the delete aspiant and prof_aspirant for data record and to keep track of the right registration no

                    // await trx(ETables.ASPIRANT)
                    //     .where("id", id)
                    //     .andWhere("email", email)
                    //     .delete();
                } else {
                    // await trx(ETables.PROF_ASPIRANT)
                    //     .where("id", id)
                    //     .andWhere("email", email)
                    //     .delete();
                }
            });
        } catch (error) {
            return this.catchError(error);
        }
    }

    public DBCreateEnrollmentV2 = async (user_id: number, common_data: { stream: Stream, entry_session: number, student_id: number, programme_id: number, session_id: number, diet_id: number }, programme_duration: string) => {
        try {
            await knex.transaction(async trx => {
                let data;
                let columns;

                if (common_data.stream === Stream.FULL) {
                    columns = [
                        'jamb_reg',
                        'o_level_path', 'o_level_mimetype', 'jamb_result_path', 'jamb_result_mimetype',
                        'adm_letter_path', 'adm_letter_mimetype'
                    ];
                } else if (common_data.stream === Stream.PART) {
                    columns = [
                        'o_level', 'o_level_mimetype',
                        'passport_path', 'passport_mimetype'
                    ];
                } else {
                    columns = [
                        'qualification', 'workplace', 'years_of_exp',
                        'certificate_mimetype', 'certificate_path'
                    ]
                }

                if (!columns) throw new Error();

                if (common_data.stream === Stream.FULL || common_data.stream === Stream.PART) {
                    data = await trx.select(columns)
                        .from(ETables.ASPIRANT)
                        .where("id", user_id)
                        .first();

                } else {
                    data = await trx.select(columns)
                        .from(ETables.PROF_ASPIRANT)
                        .where("id", user_id)
                        .first();
                }

                const completion_date = getCompletionDateV2(programme_duration);
                const enrollment_date = new Date().toISOString().split('T')[0];
                console.log({enrollment_date, completion_date})
                await trx(ETables.ENROLLMENT).insert({ ...common_data, ...data, completion_date, enrollment_date });

                await trx(ETables.ASPIRANT).where("id", user_id).delete();
            });
        } catch (error) {
            return this.catchError(error);
        }
    }
}



@Service()
export class AspirantValidatorAdapter extends BaseAdapter {
    constructor() {
        super()
    }

    public DBCheckApplicationEmail = async (email: string) => {
        try {
            const aspirant = await knex.select(['email', 'status']).from(ETables.PROF_ASPIRANT).where("email", email).first();

            const student = await knex.select(['email', 'status']).from(ETables.STUDENT).where("email", email).first();

            return aspirant || student;
        } catch (error) {
            return this.catchError(error);
        }
    }

    public DBGetProfAspirant = async (email: string) => {
        try {
            const aspirant = await knex.select('*')
                .from({ aspi: ETables.PROF_ASPIRANT })
                .join({ prog: ETables.PROGRAMME }, 'prog.id', 'aspi.programme_id')
                .where("email", email).first();

            return aspirant;
        } catch (error) {
            return this.catchError(error);
        }
    }

    public DBGetAcceptedProfAspirant = async (email: string) => {
        try {
            const aspirant = await knex.select('*')
                .from({ aspi: ETables.PROF_ASPIRANT })
                .join({ prog: ETables.PROGRAMME }, 'prog.id', 'aspi.programme_id')
                .where("email", email).andWhere('status', 'accepted').first();

            return aspirant;
        } catch (error) {
            return this.catchError(error);
        }
    }

    // NID

    public DBCheckScreeningEmail = async (email: string) => {
        try {
            const aspirant = await knex.select('*').from(ETables.ASPIRANT).where("email", email).first();

            const student = await knex.select(['email', 'status']).from(ETables.STUDENT).where("email", email).first();

            return aspirant || student;
        } catch (error) {
            return this.catchError(error);
        }
    }

    public DBCheckJambReg = async (jamb_reg: string) => {
        try {
            const aspirant = await knex.select('*').from(ETables.ASPIRANT).where("jamb_reg", jamb_reg).first();

            return aspirant;
        } catch (error) {
            return this.catchError(error);
        }
    }

    public DBGetAcceptedAspirant = async (jamb_reg: string) => {
        try {
            const aspirant = await knex.select('*')
                .from(ETables.ASPIRANT)
                .where("jamb_reg", jamb_reg).andWhere('status', 'accepted').first();

            return aspirant;
        } catch (error) {
            return this.catchError(error);
        }
    }

    public DBGetAspirant = async (jamb_reg: string) => {
        try {
            const aspirant = await knex.select('*')
                .from(ETables.ASPIRANT)
                .where("jamb_reg", jamb_reg).first();

            return aspirant;
        } catch (error) {
            return this.catchError(error);
        }

    }
    public DBGetAspirantByEmail = async (email: string) => {
        try {
            const aspirant = await knex.select('*')
                .from(ETables.ASPIRANT)
                .where("email", email).first();

            return aspirant;
        } catch (error) {
            return this.catchError(error);
        }
    }
}

export default AspirantAdapter;