import { Router } from "express";
import { Route } from "../../Interfaces/router";
import { Inject, Service } from "typedi";
import AspirantValidations from "../../Validations/Aspirant";
import AspirantController from "../../Controller/Aspirant";
import Injector from "../../Middleware/injector";
import FeeValidation from "../../Validations/Fee";
import FeeController from "../../Controller/Fee";
import { AuthMiddleware, setUploadFolder } from "../../Middleware";
// import { fileUploadErrorHandler, multerUpload } from "../../Utils/multer/index old_v2";
import { trackUploadedFiles, fileUploadErrorHandler, multerUpload } from "../../Utils/multer";
import { NID_Stream } from "../../types/enum";

@Service()
class AspirantRouter implements Route {
    public path: string;
    public router: Router;

    constructor(
        @Inject() private readonly aspirantValidation: AspirantValidations,
        private readonly aspirantController: AspirantController,
        private readonly injector: Injector,
        private readonly feeValidator: FeeValidation,
        private readonly feeController: FeeController,
        private readonly authMiddleware: AuthMiddleware
    ) {
        this.path = '/aspirant'
        this.router = Router()

        this.initializeRoutes()
    }

    private initializeRoutes () {
        this.router.get('/test', this.aspirantController.test);

        this.router.get('/profile', this.authMiddleware.isAspirantAuthenticatedV2(), this.aspirantController.getProfile)
        this.router.get('/programmes', this.aspirantController.getProfProgrammes);
        this.router.get('/check_application_status/:programme_name', this.injector.injectCurrentSessionAndDiet, this.aspirantValidation.validateCheckApplicationStatus, this.aspirantController.getApplication);
        this.router.get('/ID_card', this.authMiddleware.isAspirantAuthenticatedV2(), this.injector.injectCurrentSessionAndDiet, this.feeController.getIDCardFee)

        this.router.post('/professional', this.aspirantValidation.validateProfSignup, this.aspirantController.signup);
        this.router.get('/professional/email/verify', this.aspirantValidation.emailVerificationValidator, this.injector.injectCurrentSessionAndDiet, this.aspirantController.verifyEmail);
        this.router.post('/professional/:stream/form/pay', this.injector.injectCurrentSessionAndDiet, this.feeValidator.validateFormPayment, this.feeController.payProfFormFee);
        // this.router.post('/professional/:stream/form/verify', this.injector.injectCurrentSessionAndDiet, this.feeValidator.validateFeeVerification, this.feeController.verifyFormPayment);
        this.router.post('/professional/login', this.aspirantValidation.validateLogin, this.injector.injectCurrentSessionAndDiet, this.aspirantController.login);
        this.router.post('/professional/:stream/apply', this.authMiddleware.isProfAspirantAuthenticated, this.aspirantValidation.validateApplication, this.aspirantController.apply);
        this.router.get('/professional/:stream/acceptance', this.authMiddleware.isProfAspirantAuthenticated, this.injector.injectCurrentSessionAndDiet, this.feeValidator.validateGetProfAcceptanceFee, this.feeController.getAcceptanceFee)
        this.router.get('/professional/:stream/acceptance/pay', this.authMiddleware.isProfAspirantAuthenticated, this.injector.injectCurrentSessionAndDiet, this.feeValidator.validateAcceptancePayment, this.feeController.payAcceptanceFee);
        // this.router.post('/professional/:stream/acceptance/verify', this.authMiddleware.isProfAspirantAuthenticated, this.injector.injectCurrentSessionAndDiet, this.feeValidator.validateFeeVerification, this.feeController.verifyAcceptanceFee);
        this.router.get('/professional/:stream/school_fee', this.authMiddleware.isProfAspirantAuthenticated, this.injector.injectCurrentSessionAndDiet, this.feeValidator.validateGetSchoolFee, this.feeController.getSchoolFee)
        this.router.get('/professional/:stream/school_fee/pay', this.authMiddleware.isProfAspirantAuthenticated, this.injector.injectCurrentSessionAndDiet, this.feeValidator.validatePaySchoolFeeAndIDCard, this.feeController.paySchoolFeeAndIDCard); // paid with the ID card
        // this.router.post('/professional/:stream/school_fee/verify', this.authMiddleware.isProfAspirantAuthenticated, this.injector.injectCurrentSessionAndDiet, this.feeValidator.validateSchoolFeeAndIDCardVerification, this.feeController.verifySchoolFeeAndIDCard);
        this.router.get('/professional/:stream/acceptance_school_fee/pay', this.authMiddleware.isProfAspirantAuthenticated, this.injector.injectCurrentSessionAndDiet, this.feeValidator.validatePayAcceptanceSchoolFeeAndIDCard, this.feeController.payAcceptanceSchoolFeeAndIDCard); // paid with the ID card
        // this.router.post('/professional/:stream/acceptance_school_fee/verify', this.authMiddleware.isProfAspirantAuthenticated, this.injector.injectCurrentSessionAndDiet, this.feeValidator.validateAcceptanceSchoolFeeAndIDCardVerification, this.feeController.verifyAcceptanceSchoolFeeAndIDCard);

        // this.router.get('/email', this.aspirantController.resendVerificationCode);

        this.router.get('/putme', this.injector.injectCurrentSessionAndDiet, this.feeController.getPUTMEFee);
        // register and pay
        this.router.post('/full/register', multerUpload.fields([{name: 'o_level', maxCount: 1}, {name: 'jamb_result', maxCount: 1}]), this.injector.injectCurrentSessionAndDiet, this.aspirantValidation.validatePUTMEScreening, this.aspirantController.register);
        // this.router.post('/register/PUTME/verify', this.injector.injectCurrentSessionAndDiet, this.feeValidator.validateFeeVerification, this.feeController.verifyPUTMEPayment)
        this.router.post('/full/login', this.aspirantValidation.validateNIDLogin, this.aspirantController.nidLogin)
        // upload admission letter
        this.router.patch('/admission', multerUpload.single('adm_letter'), this.authMiddleware.isAspirantAuthenticatedV2(), this.aspirantValidation.validateUploadAdmissionLetter, this.aspirantController.uploadAdmissiionLetter);
        this.router.get('/acceptance', this.authMiddleware.isAspirantAuthenticatedV2(NID_Stream.FULL), this.injector.injectCurrentSessionAndDiet, this.feeValidator.validateGetAcceptanceFee, this.feeController.getAcceptanceFee)
        this.router.get('/acceptance/pay', this.authMiddleware.isAspirantAuthenticatedV2(NID_Stream.FULL), this.injector.injectCurrentSessionAndDiet, this.feeValidator.validateAcceptancePayment, this.feeController.payAcceptanceFee)
        // this.router.post('/acceptance/verify', this.authMiddleware.isAspirantAuthenticatedV2(NID_Stream.FULL), this.injector.injectCurrentSessionAndDiet, this.feeValidator.validateFeeVerification, this.feeController.verifyAcceptanceFee)
        this.router.get('/school_fee', this.authMiddleware.isAspirantAuthenticatedV2(NID_Stream.FULL), this.injector.injectCurrentSessionAndDiet, this.feeValidator.validateGetAcceptanceFee, this.feeController.getSchoolFee)
        this.router.get('/school_fee/pay', this.authMiddleware.isAspirantAuthenticatedV2(NID_Stream.FULL), this.injector.injectCurrentSessionAndDiet, this.feeValidator.validatePaySchoolFeeAndIDCard, this.feeController.paySchoolFeeAndIDCard)
        // this.router.post('/school_fee/verify', this.authMiddleware.isAspirantAuthenticatedV2(NID_Stream.FULL), this.injector.injectCurrentSessionAndDiet, this.feeValidator.validateSchoolFeeAndIDCardVerification, this.feeController.verifySchoolFeeAndIDCard);
        this.router.get('/acceptance_school_fee/pay', this.authMiddleware.isAspirantAuthenticatedV2(NID_Stream.FULL), this.injector.injectCurrentSessionAndDiet, this.feeValidator.validatePayAcceptanceSchoolFeeAndIDCard, this.feeController.payAcceptanceSchoolFeeAndIDCard); // paid with the ID card
        // this.router.post('/acceptance_school_fee/verify', this.authMiddleware.isAspirantAuthenticatedV2(NID_Stream.FULL), this.injector.injectCurrentSessionAndDiet, this.feeValidator.validateAcceptanceSchoolFeeAndIDCardVerification, this.feeController.verifyAcceptanceSchoolFeeAndIDCard);
        
        // PART TIME
        this.router.get('/part/application_fee', this.injector.injectCurrentSessionAndDiet, this.feeController.getPartApplicationFee);
        // this.router.post('/part/register', setUploadFolder, multerUpload.fields([{name: 'o_level', maxCount: 1}, {name: 'passport', maxCount: 1}]), this.injector.injectCurrentSessionAndDiet, this.aspirantValidation.validatePartTimeApplication, this.aspirantValidation.validateUpload, this.aspirantController.partTimeApplication);
        this.router.post('/part/register', trackUploadedFiles, multerUpload.fields([{name: 'o_level', maxCount: 1}, {name: 'passport', maxCount: 1}]), this.injector.injectCurrentSessionAndDiet, this.aspirantValidation.validatePartTimeApplication, this.aspirantValidation.validateUpload, this.aspirantController.partTimeApplication);
        this.router.post('/part/login', this.aspirantValidation.validatePartLogin, this.aspirantController.partLogin)
        this.router.get('/part/application_form', this.authMiddleware.isAspirantAuthenticatedV2(NID_Stream.PART), this.aspirantController.generateApplicationForm)

        this.router.use(fileUploadErrorHandler)
    }
}

export default AspirantRouter;