import { addWeeks, addYears, format } from "date-fns";
import knex from "../Database/adapters";
import { ETables } from "../types/db.types";
import bcrypt from 'bcryptjs';
import { Stream } from "../types/enum";
import sharp from "sharp";

export const pinGenerator = (l: number): string => {
    const p = l - 1;
    const value = Math.floor(Math.pow(10, p) + Math.random() * (9 * Math.pow(10, p)));

    return value.toString();
}

export const fileStore = new Map();

export const encodeLastName = async (last_name: string) => {
    const salt = await bcrypt.genSalt(10);
    return bcrypt.hashSync(last_name.toLowerCase(), salt);
}

export const generateRegNo = async (programme: any, diet: any) => {
    // const programme = await knex.select('*').from(ETables.PROGRAMME).where("id", programme_id).first();

    const students = await knex.select(['email', 'adm_no']).from(ETables.STUDENT);

    let prog;
    switch (programme.stream) {
        case Stream.PART:
            prog = 'PAR'
            break;
        case Stream.FULL:
            prog = 'FUL'
            break;
        case 'basic':
            prog = 'BAS'
            break;
        case 'advanced':
            prog = 'ADV'
            break;
        default:
            break;
    }

    if (!students.length) return `TNM/${prog}/${diet.number}${diet.year % 100}/0001`

    const nos: number[] = [];
    const stds = students.map(s => s.adm_no)
    stds.map(no => {
        const splitted = no.split('/').pop();
        console.log(splitted, 'splitted')
        // let num = splitted[splitted.length - 1];

        nos.push(+splitted);
    });

    let last = Math.max(...nos) + 1;
    let formattedLast = String(last).padStart(4, '0');
    return `TNM/${prog}/${diet.number}${diet.year % 100}/${formattedLast}`;
}

export const generateTransactionID = (): number => {
    const rand = Math.floor(Math.pow(10, 2) + Math.random() * 10)
    const t = Date.now();

    return rand + t;
}

export function excelSerialDateToJSDate(serialDate: number) {
    // Excel serial date starts on January 1, 1900. JavaScript's Date object starts on January 1, 1970.
    // Additionally, Excel incorrectly treats February 29, 1900, as a valid date, which it is not
    // because 1900 is not a leap year. Hence, any serial date after 59 is off by one. 
    // To compensate, we subtract 1 for dates after February 28, 1900.
    const excelEpoch = new Date(1900, 0, -1);
    const jsDate = new Date(excelEpoch.getTime() + serialDate * 86400000);

    return jsDate;
}


export const drawTable = (doc: PDFKit.PDFDocument, data: any) => {
    const tableTop = 100;
    const columnWidths = [200, 300]; // Adjust based on your needs
    const rowHeight = 20;
    let currentRowHeight = tableTop;

    // Draw headers
    doc.fontSize(12).fillColor('black').text('Name', 50, tableTop);
    doc.text('Courses', 50 + columnWidths[0], tableTop);

    // Draw each row
    data.forEach((row: any, i: number) => {
        currentRowHeight += rowHeight;
        doc.fontSize(10).fillColor('black').text(row.name, 50, currentRowHeight);
        doc.text(row.courses.join(', '), 50 + columnWidths[0], currentRowHeight);
    });
}

export const calculateCharges = (amount: number) => {
    let fee: number;
    if (amount <= 2500) return fee = 1.5 / 100 * amount
    else {
        fee = (1.5 / 100 * amount) + 100
        if (fee > 2000) return 2000
        return fee;
    }
}

export const generateApplicationNo = async () => {
    // const programme = await knex.select('*').from(ETables.PROGRAMME).where("id", programme_id).first();

    const aspirants = await knex.select(['email', 'reg_no']).from(ETables.ASPIRANT);
    const prof_aspirants = await knex.select(['email', 'reg_no']).from(ETables.PROF_ASPIRANT);

    const combined = aspirants.concat(prof_aspirants);

    const currYear = new Date().getFullYear();
    if (!combined.length) return `TNM${currYear}${currYear + 1}/0001`

    const nos: number[] = [];
    const applicants = combined.map(s => s.reg_no)
    applicants.map(no => {
        const splitted = no.split('/').pop();
        console.log(splitted, 'splitted')
        // let num = splitted[splitted.length - 1];

        nos.push(+splitted);
    });

    let last = Math.max(...nos) + 1;
    let formattedLast = String(last).padStart(4, '0');
    return `TNM${currYear}${currYear + 1}/${formattedLast}`
}

export const getCompletionDate = async (duration: string) => {
    const [number, timeframe] = duration.split(' ');
    const num = parseInt(number, 10);

    if (isNaN(num)) {
        throw new Error('Invalid duration number');
    }

    const currentDate = new Date();

    let completion_date: Date;

    if (timeframe === 'weeks') {
        completion_date = new Date(currentDate);
        completion_date.setDate(currentDate.getDate() + num * 7);
    } else if (timeframe === 'years') {
        completion_date = new Date(currentDate);
        completion_date.setFullYear(currentDate.getFullYear() + num);
    } else {
        throw new Error('Invalid timeframe. Only "weeks" or "years" are supported.');
    }

    // Format the completion date in the "YYYY-MM-DD" format
    const formattedDate = completion_date.toISOString().split('T')[0];

    return formattedDate;
};

export const getCompletionDateV2 = (duration: string) => {
    const [number, timeframe] = duration.split(' ');

    const num = parseInt(number, 10);
    if (isNaN(num)) {
        throw new Error('Invalid duration number');
    }

    const currentDate = new Date();

    let completionDate: Date;

    if (timeframe === 'weeks') {
        completionDate = addWeeks(currentDate, num);
    } else if (timeframe === 'years') {
        completionDate = addYears(currentDate, num);
    } else {
        throw new Error('Invalid timeframe. Only "weeks" or "years" are supported.');
    }

    const formattedDate = format(completionDate, 'yyyy-MM-dd');

    return formattedDate;
};

export const resizeImage = async (imagePath: string, width: number, height: number, shouldResize = true) => {
    if (!shouldResize) {
        const originalImageBuffer = await sharp(imagePath).jpeg({ quality: 70 }).toBuffer();
        return originalImageBuffer.toString('base64');
    }

    const resizedImageBuffer = await sharp(imagePath)
        .resize(width, height)
        .jpeg({ quality: 70 })
        .toBuffer();
    return resizedImageBuffer.toString('base64');
};
