import { Service } from "typedi";
import Validator from "..";
import * as bcrypt from "bcryptjs";
import { AdminValidatorAdapter } from "../../Database/adapters/adminAdapter";
import { CourseValidatorAdapter } from "../../Database/adapters/courseAdapter";
import { Category, Stream, Status, Course_Type } from "../../types/enum";
import { AspirantValidatorAdapter } from "../../Database/adapters/aspirantAdapter";

@Service()
class AdminValidator extends Validator {
  constructor(
    private readonly adminValAdapter: AdminValidatorAdapter,
    private readonly courseValAdapter: CourseValidatorAdapter,
    private readonly aspirantValAdapter: AspirantValidatorAdapter,
  ) {
    super()
  }

  public loginValidator = this.validate({
    email: {
      in: ["body"],
      isEmail: true,
      toLowerCase: true,
    },
    password: {
      in: ["body"],
      isString: true,
      isLength: { options: { min: 8 } },
      custom: {
        options: async (password, { req }) => {
          console.log('hERE');
          
          const user = await this.adminValAdapter.DBGetUserAndPassword(req.body.email);

          if (!user || !bcrypt.compareSync(password, user.password)) {
            throw new Error("Invalid credentials!")
          }

          req.body.id = user.id;
          req.body.role = user.role;
        }
      },
      errorMessage: 'Invalid credentials'
    }
  })

  public validateEditAdmin = this.validate({
    id: {
      in: ['params'],
      isInt: true,
      custom: {
        options: async id => {
          const valid = await this.adminValAdapter.DBGetAdmin(id);

          if(!valid) throw new Error("Admin doesn't exist");
        }
      }
    },
    role_id: {
      in: ['body'],
      isInt: true,
      optional: true,
      custom: {
        options: async role_id => {
          const valid = await this.adminValAdapter.checkForValidRole(role_id);

          if(!valid) throw new Error('Invalid role')
        }
      }
    },
    name: {
      in: ['body'],
      isString: true,
      optional: true
    },
    email: {
      in: ['body'],
      isEmail: true,
      optional: true,
      custom: {
        options: async (email, { req }) => {
          const exists = await this.adminValAdapter.DBGetUserAndPassword(email);
          console.log(req.params?.id !== exists.id )
          if(+req.params?.id !== exists.id && exists) throw new Error('Admin already exists');
        }
      }
    }
  })

  public validateGetProgrammeByCategory = this.validate({
    prog_level: {
      in: ['params'],
      isString: true,
      isIn: { options: [Object.values(Stream)] },
      custom: {
        options: async (prog_level) => {
          const exist = await this.adminValAdapter.checkProgramByCategory(prog_level);

          if (!exist) throw new Error('Invalid programme')
        }
      }
    }
  })

  public validateGetCourses = this.validate({
    stream: {
      in: ['params'],
      isString: true,
      isIn: { options: [Object.values(Stream)] }
    }
  })

  public validateGetACourse = this.validate({
    stream: {
      in: ['params'],
      isString: true,
      isIn: { options: [Object.values(Stream)] }
    },
    id: {
      in: 'params',
      isInt: true,
      custom: {
        options: async id => {
          const exist = await this.courseValAdapter.DBGetCourseById(id);

          if(!exist) throw new Error('Invalid course');
        }
      }
    }
  })

  public validateAssignStaff = this.validate({
    stream: {
      in: ['params'],
      isString: true,
      isIn: { options: [Object.values(Stream)] }
    },
    id: {
      in: 'params',
      isInt: true,
      custom: {
        options: async (id, { req }) => {
          const exist = await this.courseValAdapter.DBCourseAlreadyAssignedForTheSession(id, req.body.session.id);

          if(exist) throw new Error('Course already asigned for the selected session');
        }
      }
    },
    admin_ids: {
      in: 'body',
      isArray:  true,
      isInt: true,
      custom: {
        options: async admin_ids => {
          if(new Set(admin_ids).size !== admin_ids.length) throw new Error('Duplicate id found');

          for(let admin_id of admin_ids) {
            const exists = await this.adminValAdapter.DBGetAdmin(admin_id);
  
            if(!exists) throw new Error('Invalid admin')
          }
        }
      }
    }
  })

  public validateGetAssignedStaff = this.validate({
    stream: {
      in: ['params'],
      isString: true,
      isIn: { options: [Object.values(Stream)] }
    },
    id: {
      in: 'params',
      isInt: true,
      custom: {
        options: async id => {
          const exist = await this.courseValAdapter.DBGetCourseById(id);

          if(!exist) throw new Error('Invalid course');
        }
      }
    }
  })

  public validateGetApplications = this.validate({
    stream: {
      in: ['params'],
      isString: true,
      isIn: { options: [ Object.values(Stream)]},
      custom: {
        options: async stream => {
          const valid = await this.adminValAdapter.CheckProgrammeByStream(stream);

          if (!valid) throw new Error('Invalid programme');
        }
      }
    }
  })

  public validateGetAnApplication = this.validate({
    stream: {
      in: ['params'],
      isString: true,
      isIn: { options: [ Object.values(Stream)]}
    },
    id: {
      in: ['params'],
      isInt: true,
      custom: {
        options: async (id, { req }) => {
          const valid = await this.adminValAdapter.DBCheckApplication(id, req.params?.stream);
          
          if(!valid) throw new Error('Applicannt does not exist');
        }
      }
    }
  })

  public validateApplicationReview = this.validate({
    id: {
      in: ['params'],
      isInt: true,
      custom: {
        options: async (id, { req }) => {
          const valid = await this.adminValAdapter.DBCheckApplication(id, req.params?.stream);

          if (!valid) throw new Error('Application does not exist');
        }
      }
    },
    status: {
      in: ['body'],
      isIn: { options: ['accepted', 'rejected'] }
    },
    stream: {
      in: 'params',
      isIn: { options: [ Object.values(Stream) ]},
      custom: {
        options: async (stream, { req }) => {
          const valid = await this.adminValAdapter.CheckProgrammeByStream(stream);

          if (!valid) throw new Error('Invalid programme');

          req.body.programme_id = valid.id
        }
      }
    },
    comment: {
      in: ['body'],
      isString: true
    }
  })

  public validateGetCourseResult = this.validate({
    id: {
      in: 'params',
      isInt: true,
      custom: {
        options: async id => {
          const exist = await this.courseValAdapter.DBGetCourseById(id);

          if(!exist) throw new Error('Invalid course');
        }
      }
    },
    session_id: {
      in: ['params'],
      isInt: true,
      custom: {
        options: async session_id => {
          const valid = this.adminValAdapter.IsAValidSession(session_id);

          if(!valid) throw new Error('Invalid session');
        }
      }
    }
  })

  public validateDownloadFile = this.validate({
    file_path: {
      in: 'params',
      isString: true,
      custom: {
        options: async file_path => {
          if(file_path ===  null || file_path === undefined || file_path === 'null') throw new Error('Invalid file')
        }
      }
    }
  })








































  public validateSetFee = this.validate({
    name: {
      in: ['body'],
      isString: true
    },
    amount: {
      in: ['body'],
      isFloat: true
    },
    prog_level: {
      in: ["body"],
      isString: true,
      isIn: { options: [Object.values(Stream)] },
      optional: true
    },
    programme_id: {
      in: ['body'],
      isInt: true,
      optional: true,
      custom: {
        options: async (programme_id) => {
          const programme_exist = await this.adminValAdapter.DoesProgrammeExist(programme_id);

          if (!programme_exist) throw new Error('')
        }
      }
    },
    session_id: {
      in: ['body'],
      isInt: true,
      custom: {
        options: async (session_id, { req }) => {
          const isValid = await this.adminValAdapter.IsAValidSession(session_id);

          if (!isValid) throw new Error('Invalid session provided');

          req.body = {
            ...req.body,
            session: isValid.name
          }
        }
      }
    },
    category: {
      in: ['body'],
      isString: true,
      isIn: { options: [Object.values(Category)] }
    },
    is_recurrent: {
      in: ['body'],
      isBoolean: true,
    }
  })

  public validateCreateCourse = this.validate({
    title: {
      in: ["body"],
      isString: true,
      notEmpty: true,
      custom: {
        options: async (title) => {
          // const exists = await this.courseValAdapter.DBDoesCourseTitleExist(title);

          // if (exists) throw new Error('Title already exist');
        }
      }
    },
    code: {
      in: ["body"],
      isString: true,
      isLength: { options: { min: 6 } },
      custom: {
        options: async (code) => {
          // const exists = await this.courseValAdapter.DBDoesCourseCodeExist(code);

          // if (exists) throw new Error('Duplicate course code');
        }
      }
    },
    description: {
      in: ["body"],
      isString: true,
      notEmpty: true,
    },
    unit: {
      in: ["body"],
      isInt: true,
    },
    type: {
      in: ["body"],
      isString: true,
      isIn: { options: [Object.values(Course_Type)] },
    },
    prog_level: {
      in: ["body"],
      isString: true,
      isIn: { options: [Object.values(Stream)] },
    },
  })

  public validateUploadAspirant = this.validate({
    sheetName: {
      in: ['body'],
      isString: true,
      custom: {
        options: async (sheetName, { req }) => {
          if (!req.file) throw new Error('No file uploaded')

          return true
        }
      }
    },
    session_id: {
      in: ['body'],
      isInt: true,
      custom: {
        options: async (session_id, { req }) => {
          const isValid = await this.adminValAdapter.IsAValidSession(session_id);

          if (!isValid) throw new Error('Invalid session provided');

          req.body = {
            ...req.body,
            session: isValid.name
          }
        }
      }
    },
    year: {
      in: ['body'],
      isInt: true
    },
  })

  public validateViewAnAspirant = this.validate({
    jamb_reg: {
      in: ['params'],
      isString: true,
      custom: {
        options: async (jamb_reg) => {
          // const exist = await this.aspirantValAdapter.DBGetAspirant(jamb_reg)

          // if (!exist) throw new Error('Aspirant does not exist');
        }
      }
    }
  })

  public validatePutmeResultUpload = this.validate({
    type: {
      in: 'body',
      isString: true,
    },
    session_id: {
      in: ['body'],
      isInt: true,
      custom: {
        options: async (session_id, { req }) => {
          const isValid = await this.adminValAdapter.IsAValidSession(session_id);

          if (!isValid) throw new Error('Invalid session provided');

          req.body = {
            ...req.body,
            session: isValid.name
          }
        }
      }
    },
    year: {
      in: ['body'],
      isInt: true
    },
    sheetName: {
      in: ['body'],
      isString: true,
      custom: {
        options: async (sheetName, { req }) => {
          if (!req.file) throw new Error('No file uploaded')

          return true
        }
      }
    }
  })

  public validateAspirantsStatus = this.validate({
    status: {
      in: ['query'],
      isString: true,
      isIn: { options: [Object.values(Status)] }
    },
    jamb_regs: {
      in: ['body'],
      isArray: { options: { min: 1 } },
      notEmpty: true,
      // isString: true,
      custom: {
        options: async (jamb_regs, { req }) => {
          if (new Set(jamb_regs).size !== jamb_regs.length) throw new Error('The courses must be unique');

          for (let reg of jamb_regs) {
            // const exist = await this.aspirantValAdapter.DBGetAspirant(reg);

            // if (!exist) throw new Error('Aspirant does not exist');

            // if (!exist.grade) throw new Error(`One of the aspirant doesn't have a grade`);
          }

        }
      }
    }
  })

  public validateGetStudents = this.validate({
    programme_id: {
      in: ['params'],
      isInt: true,
      custom: {
        options: async (programme_id) => {
          const exist = await this.adminValAdapter.DoesProgrammeExist(programme_id);

          if (!exist) throw new Error('Invalid programme');
        }
      }
    }
  })
}

export default AdminValidator;
