import * as bcrypt from 'bcryptjs';
import Validator from "..";
import { AspirantValidatorAdapter } from "../../Database/adapters/aspirantAdapter";
import { decodeProfAspirantFormToken } from "../../Utils/jwt";
import { Category, Gender, PROF_Stream, PROGRAMMES, Stream } from "../../types/enum";
import { StudentValidatorAdapter } from '../../Database/adapters/studentAdapter';
import { Service } from 'typedi';
import { FeeValidatorAdapter } from '../../Database/adapters/feeAdapter';
import { NextFunction, Request, Response } from 'express';
import { badRequest } from '../../Utils/api_response';

@Service()
class AspirantValidation extends Validator {
    constructor(
        private readonly aspirantValAdapter: AspirantValidatorAdapter,
        private readonly studentValAdapter: StudentValidatorAdapter,
        private readonly feeValAdapter: FeeValidatorAdapter
    ) {
        super()
    }

    public validateProfSignup = this.validate({
        email: {
            in: ["body"],
            isEmail: true,
            toLowerCase: true,
            custom: {
                options: async (email) => {
                    const aspirant = await this.aspirantValAdapter.DBCheckApplicationEmail(email);

                    if (aspirant) throw new Error("Email already in use!")
                }
            }
        },
        // password: {
        //     in: ["body"],
        //     isString: true,
        //     isLength: { options: { min: 8 } },
        // },
        first_name: {
            in: ["body"],
            isString: true,
            notEmpty: true,
        },
        last_name: {
            in: ["body"],
            isString: true,
            notEmpty: true,
        },
        middle_name: {
            in: ["body"],
            isString: true,
            // optional: true
        },
        phone: {
            in: ["body"],
            isMobilePhone: true,
            // optional: true,
            errorMessage: 'Invalid phone number'
        },
        stream: {
            in: ['body'],
            isIn: { options: [Object.values(PROF_Stream)] },
        }
    })

    public emailVerificationValidator = this.validate({
        // email: {
        //     in: ["body"],
        //     isEmail: true,
        //     toLowerCase: true,
        // },
        token: {
            in: ["query"],
            isString: true,
            custom: {
                options: async (token, { req }) => {
                    try {
                        const { email, first_name, last_name, middle_name, stream } = decodeProfAspirantFormToken(token);

                        const str = stream as any;

                        if (!(Object.values(PROF_Stream).includes(stream))) throw new Error('Invalid stream provided');

                        const openToAdmission = await this.feeValAdapter.DBIsProgrammeOpenToAdmission(str);

                        if (!openToAdmission) throw new Error('This programme is not open to appication at the moment, please check back later')

                        req.body['aspirant'] = {
                            email,
                            stream,
                            first_name,
                            last_name,
                            middle_name
                        }
                        // console.log(req.body.aspirant)
                    } catch (error) {
                        throw new Error("Verification code expired!")
                    }
                }
            }
        },
        // stream: {
        //     in: 'body',
        //     isString: true,
        //     isIn: { options: [Object.values(PROF_Stream)] }
        // }
    })

    public validateLogin = this.validate({
        email: {
            in: ["body"],
            isEmail: true,
            toLowerCase: true,
        },
        password: {
            in: ["body"],
            isString: true,
            notEmpty: true,
            // isLength: { options: { min: 8 } },
            custom: {
                options: async (password, { req }) => {

                    const aspirant = await this.aspirantValAdapter.DBGetProfAspirant(req.body.email);

                    if (!aspirant || !bcrypt.compareSync(password, aspirant.password)) throw new Error("Invalid credentials!");
                }
            }
        }
    })

    public validateCheckApplicationStatus = this.validate({
        programme_name: {
            in: ["params"],
            isString: true,
            isIn: { options: [Object.values(PROGRAMMES)] }
        },
    })

    public validateApplication = this.validate({
        middle_name: {
            in: ["body"],
            isString: true,
        },
        email: {
            in: ["body"],
            isEmail: true,
            toLowerCase: true,
            custom: {
                options: async (email, { req }) => {
                    const { auth_user } = req.body;
                    // console.log(req.body, 'req.body')
                    const aspirant = await this.aspirantValAdapter.DBCheckApplicationEmail(email);

                    // check if paid
                    const form_fee = await this.feeValAdapter.DBGetFeeByNameAndCategory('Form', Category.PROFESSIONAL);

                    const has_paid_form_fee = await this.feeValAdapter.DBHasPaidFee(form_fee.id, auth_user.id);

                    if (!has_paid_form_fee) throw new Error('You not eligible to this resource, kindly pay the application form to continue');

                    if (aspirant.status) throw new Error('You can only fill the application form once.')

                    if (aspirant.email !== auth_user.email) throw new Error("Invalid email");


                    // if (aspirant.stream !== auth_user.stream) throw new Error('Invalid stream');
                }
            }
        },
        DOB: {
            in: ['body'],
            isString: true,
            isISO8601: { errorMessage: "invalid iso" },
        },
        gender: {
            in: 'body',
            isIn: { options: [Object.values(Gender)] }
        },
        stream: {
            in: ['params'],
            isIn: { options: [Object.values(PROF_Stream)] },
        },
        qualification: {
            in: 'body',
            isString: true
        },
        workplace: {
            in: 'body',
            isString: true
        },
        years_of_exp: {
            in: 'body',
            isInt: true
        },
    })

    // NID

    public validatePUTMEScreening = this.validate({
        first_name: {
            in: ["body"],
            isString: true,
            notEmpty: true,
        },
        last_name: {
            in: ["body"],
            isString: true,
            notEmpty: true,
        },
        middle_name: {
            in: ["body"],
            isString: true,
            optional: true
        },
        email: {
            in: ["body"],
            isEmail: true,
            toLowerCase: true,
            custom: {
                options: async (email) => {
                    const aspirant = await this.aspirantValAdapter.DBCheckScreeningEmail(email);

                    if (aspirant) throw new Error("Email already exits!")
                }
            }
        },
        // stream: {
        //     in: ['body'],
        //     isIn: { options: [Object.values(PROF_Stream)] },
        // },
        jamb_reg: {
            in: 'body',
            isString: true,
            isLength: { options: { min: 14 } },
            custom: {
                options: async (jamb_reg, { req }) => {
                    const exist = await this.aspirantValAdapter.DBCheckJambReg(jamb_reg);

                    if (exist) throw new Error('Account already exist')
                    console.log(req.file, 'file')
                    console.log(req.files, 'files')
                    const { o_level, jamb_result } = req.files;

                    if (!o_level || !jamb_result) throw new Error('Please upload the required documents');
                }
            }
        },
        DOB: {
            in: ['body'],
            isString: true,
            isISO8601: { errorMessage: "invalid iso" },
        },
        gender: {
            in: 'body',
            isIn: { options: [Object.values(Gender)] }
        },
    })

    public validateNIDLogin = this.validate({
        jamb_reg: {
            in: 'body',
            isString: true,
            isLength: { options: { min: 14 } },
        },
        password: {
            in: ['body'],
            isString: true,
            custom: {
                options: async (password, { req }) => {
                    const aspirant = await this.aspirantValAdapter.DBGetAspirant(req.body.jamb_reg);

                    if (!aspirant || !bcrypt.compareSync(password, aspirant.password)) throw new Error("Invalid credentials!")
                }
            }
        }
    })

    public validateUploadAdmissionLetter = async (req: Request, res: Response, next: NextFunction) => {
        try {
            // const { buffer, mimetype } = req.file!

            // if (!buffer) throw new Error('Kindly upload your admission letter')
            if (!req.file) throw new Error('Kindly upload your admission letter')

            return next();
        } catch (error) {
            console.log(error)
            return badRequest(res, null, "Something went wrong");
        }
    }

    // public validateUpload = () = async (req: Request, res: Response, next: NextFunction) => {
    public validateUpload = async (req: Request, res: Response, next: NextFunction) => {
        try {
            const files = req.files as { [fieldname: string]: Express.Multer.File[] };
            const { o_level, passport } = files;

            // Check if files are uploaded
            if (!o_level || !passport || !o_level[0] || !passport[0]) throw new Error('Kindly upload the necessary files')

            // const allowedFileTypes = ['application/pdf', 'application/vnd.openxmlformats-officedocument.wordprocessingml.document', 'application/msword'];

            // // Check the MIME types of the uploaded files
            // const o_levelMimetype = o_level[0].mimetype;
            // const passportMimetype = passport[0].mimetype;

            // if (!allowedFileTypes.includes(o_levelMimetype) || !allowedFileTypes.includes(passportMimetype)) {
            //     throw new Error("Invalid file type");
            // }

            return next();
        } catch (error) {
            console.log(error);
            return badRequest(res, null, "Something went wrong");
        }
    };


    // NID PART TIME
    public validatePartTimeApplication = this.validate({
        first_name: {
            in: ["body"],
            isString: true,
            notEmpty: true,
        },
        last_name: {
            in: ["body"],
            isString: true,
            notEmpty: true,
        },
        middle_name: {
            in: ["body"],
            isString: true,
            optional: true
        },
        email: {
            in: ["body"],
            isEmail: true,
            toLowerCase: true,
            custom: {
                options: async (email, { req }) => {
                    const aspirant = await this.aspirantValAdapter.DBCheckScreeningEmail(email);

                    if (aspirant) throw new Error("Email already exits!")
                }
            }
        },
        // stream: {
        //     in: ['body'],
        //     isIn: { options: [Object.values(PROF_Stream)] },
        // },
        DOB: {
            in: ['body'],
            isString: true,
            isISO8601: { errorMessage: "invalid iso" },
        },
        gender: {
            in: 'body',
            isIn: { options: [Object.values(Gender)] }
        },
        state_of_origin: {
            in: 'body',
            isString: true
        },
        lga: {
            in: 'body',
            isString: true
        },
        phone: {
            in: 'body',
            isMobilePhone: { options: 'any' }
        },
        address: {
            in: 'body',
            isString: true
        },
        // manual_o_level
    })

    public validatePartLogin = this.validate({
        email: {
            in: 'body',
            isString: true,
            isEmail: true,
            toLowerCase: true
        },
        password: {
            in: ['body'],
            isString: true,
            custom: {
                options: async (password, { req }) => {
                    const aspirant = await this.aspirantValAdapter.DBGetAspirantByEmail(req.body.email);

                    if (!aspirant || !bcrypt.compareSync(password, aspirant.password)) throw new Error("Invalid credentials!")
                }
            }
        }
    })
}


export default AspirantValidation;