import bcrypt from 'bcryptjs';
import Validator from "..";
import { StudentValidatorAdapter } from "../../Database/adapters/studentAdapter";
import { Service } from 'typedi';
// import { decodeOTP } from '../../Utils/jwt';
import { Level, Semester } from '../../types/enum';
import { NextFunction, Request, Response } from 'express';
import { badRequest } from '../../Utils/api_response';
import { FeeValidatorAdapter } from '../../Database/adapters/feeAdapter';

@Service()
class StudentValidation extends Validator {
  constructor(
    private readonly studentValAdapter: StudentValidatorAdapter,
    private readonly feeValAdapter: FeeValidatorAdapter
  ) {
    super()
  }

  public loginValidator = this.validate({
    matric_no: {
      in: ["body"],
      isString: true,
      isLength: { options: { min: 4 } }
    },
    password: {
      in: ["body"],
      isString: true,
      notEmpty: true,
      custom: {
        options: async (password, { req }) => {
          const { matric_no } = req.body;
          const student = await this.studentValAdapter.DBGetStudent(matric_no);

          if (!student || !bcrypt.compareSync(password, student.password)) throw new Error("Invalid credentials!");
        }
      }
    }
  })

  public validateGetResult = this.validate({
    level: {
      in: 'query',
      isString: true,
      isIn: { options: Object.values(Level) },
      optional: true
    },
    // current_semester: {
    //   in: 'query',
    //   isBoolean: true,
    //   toBoolean: true
    // },
    semester: {
      in: 'query',
      isString: true,
      isIn: { options: Object.values(Semester) },
      optional: true,
      custom: {
        options: async (semester, { req }) => {
          // // if (!req.query?.current_semester && !semester) throw new Error("Semester can't be undefined when it's not for the current semester")
          // // else if (req.query?.current_semester && semester) throw new Error('Semester canoot be defined when current semester is enabled')
          // // else {
          //   const valid_session = await this.studentValAdapter.DBIsValidSession(+req.query?.session_id);

          //   if (!valid_session) throw new Error('Invalid session')
          // // }
        }
      }
    },
    session_id: {
      in: ['query'],
      isInt: true,
      custom: {
        options: async (session_id, { req }) => {
          if(!req.query?.level && req.query?.semester) {
            throw new Error("The evel can't be undeifned hen the semester is defined");
          } else if(!req.query?.level && +session_id !== 0) {
            throw new Error('Session_id has to be zero if level is undefined');
          } else if(session_id && req.query?.level) {
            const valid_session = await this.studentValAdapter.DBIsValidSession(session_id);

            if (!valid_session) throw new Error('Invalid session');
          }
        }
      }
    },
    diet: {
      in: 'query',
      isInt: true,
      optional: true
    }
  })

  // public validateGetExamSlip = this.validate({})

  public validateGetExamSlip = async (req: Request, res: Response, next: NextFunction) => {
    try {
      const { auth_user, session } = req.body;

      const outstandingFee = await this.feeValAdapter.DBCheckOutstandingFee(auth_user.id, session.id, auth_user.stream, true);

      if(outstandingFee) return badRequest(res, null, 'You need to clear any outstanding fee before you can access this resource');

      return next();
    } catch (error) {
      return badRequest(res, null, "Something went wrong");
    }
  }

  public validateGetInternLetter = async (req: Request, res: Response, next: NextFunction) => {
    try {
      const { auth_user, session } = req.body;

      if(auth_user.stream !== 'standard') return badRequest(res, null, 'THis is only available for advance certificate programme');

      const outstandingFee = await this.feeValAdapter.DBCheckOutstandingFee(auth_user.id, session.id, auth_user.stream, true);

      if(outstandingFee) return badRequest(res, null, 'You need to clear any outstanding fee before you can access this resource');

      return next();
    } catch (error) {
      return badRequest(res, null, "Something went wrong");
    }
  }
}

export default StudentValidation;